/*
 * Decompiled with CFR 0.152.
 */
package org.mariadb.jdbc.internal.util.pool;

import java.lang.management.ManagementFactory;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.sql.ConnectionEvent;
import javax.sql.ConnectionEventListener;
import org.mariadb.jdbc.MariaDbConnection;
import org.mariadb.jdbc.MariaDbPooledConnection;
import org.mariadb.jdbc.UrlParser;
import org.mariadb.jdbc.internal.logging.Logger;
import org.mariadb.jdbc.internal.logging.LoggerFactory;
import org.mariadb.jdbc.internal.protocol.Protocol;
import org.mariadb.jdbc.internal.util.Utils;
import org.mariadb.jdbc.internal.util.exceptions.ExceptionFactory;
import org.mariadb.jdbc.internal.util.pool.GlobalStateInfo;
import org.mariadb.jdbc.internal.util.pool.PoolMBean;
import org.mariadb.jdbc.internal.util.pool.Pools;
import org.mariadb.jdbc.internal.util.scheduler.MariaDbThreadFactory;
import org.mariadb.jdbc.util.Options;

public class Pool
implements AutoCloseable,
PoolMBean {
    private static final Logger logger = LoggerFactory.getLogger(Pool.class);
    private static final int POOL_STATE_OK = 0;
    private static final int POOL_STATE_CLOSING = 1;
    private final AtomicInteger poolState = new AtomicInteger();
    private final UrlParser urlParser;
    private final Options options;
    private final AtomicInteger pendingRequestNumber = new AtomicInteger();
    private final AtomicInteger totalConnection = new AtomicInteger();
    private final LinkedBlockingDeque<MariaDbPooledConnection> idleConnections;
    private final ThreadPoolExecutor connectionAppender;
    private final BlockingQueue<Runnable> connectionAppenderQueue;
    private final String poolTag;
    private final ScheduledThreadPoolExecutor poolExecutor;
    private final ScheduledFuture scheduledFuture;
    private GlobalStateInfo globalInfo;
    private int maxIdleTime;
    private long timeToConnectNanos;
    private long connectionTime = 0L;

    public Pool(UrlParser urlParser, int poolIndex, ScheduledThreadPoolExecutor poolExecutor) {
        this.urlParser = urlParser;
        this.options = urlParser.getOptions();
        this.maxIdleTime = this.options.maxIdleTime;
        this.poolTag = this.generatePoolTag(poolIndex);
        this.connectionAppenderQueue = new ArrayBlockingQueue<Runnable>(this.options.maxPoolSize);
        this.connectionAppender = new ThreadPoolExecutor(1, 1, 10L, TimeUnit.SECONDS, this.connectionAppenderQueue, new MariaDbThreadFactory(this.poolTag + "-appender"));
        this.connectionAppender.allowCoreThreadTimeOut(true);
        this.connectionAppender.prestartCoreThread();
        this.idleConnections = new LinkedBlockingDeque();
        int scheduleDelay = Math.min(30, this.maxIdleTime / 2);
        this.poolExecutor = poolExecutor;
        this.scheduledFuture = poolExecutor.scheduleAtFixedRate(this::removeIdleTimeoutConnection, scheduleDelay, scheduleDelay, TimeUnit.SECONDS);
        if (this.options.registerJmxPool) {
            try {
                this.registerJmx();
            }
            catch (Exception ex) {
                logger.error("pool " + this.poolTag + " not registered due to exception : " + ex.getMessage());
            }
        }
        try {
            for (int i = 0; i < this.options.minPoolSize; ++i) {
                this.addConnection();
            }
        }
        catch (SQLException sqle) {
            logger.error("error initializing pool connection", sqle);
        }
    }

    private void addConnectionRequest() {
        if (this.totalConnection.get() < this.options.maxPoolSize && this.poolState.get() == 0) {
            this.connectionAppender.prestartCoreThread();
            this.connectionAppenderQueue.offer(() -> {
                if ((this.totalConnection.get() < this.options.minPoolSize || this.pendingRequestNumber.get() > 0) && this.totalConnection.get() < this.options.maxPoolSize) {
                    try {
                        this.addConnection();
                    }
                    catch (SQLException sqle) {
                        logger.error("error adding connection to pool", sqle);
                    }
                }
            });
        }
    }

    private void removeIdleTimeoutConnection() {
        Iterator<MariaDbPooledConnection> iterator = this.idleConnections.descendingIterator();
        while (iterator.hasNext()) {
            MariaDbPooledConnection item = iterator.next();
            long idleTime = System.nanoTime() - item.getLastUsed().get();
            boolean timedOut = idleTime > TimeUnit.SECONDS.toNanos(this.maxIdleTime);
            boolean shouldBeReleased = false;
            if (this.globalInfo != null) {
                if (idleTime > TimeUnit.SECONDS.toNanos(this.globalInfo.getWaitTimeout() - 45)) {
                    shouldBeReleased = true;
                }
                if (timedOut && this.totalConnection.get() > this.options.minPoolSize) {
                    shouldBeReleased = true;
                }
            } else if (timedOut) {
                shouldBeReleased = true;
            }
            if (!shouldBeReleased || !this.idleConnections.remove(item)) continue;
            long connId = item.getConnection().getServerThreadId();
            this.totalConnection.decrementAndGet();
            this.silentCloseConnection(item);
            this.addConnectionRequest();
            if (!logger.isDebugEnabled()) continue;
            logger.debug("pool {} connection {} removed due to inactivity (total:{}, active:{}, pending:{})", this.poolTag, connId, this.totalConnection.get(), this.getActiveConnections(), this.pendingRequestNumber.get());
        }
    }

    private void addConnection() throws SQLException {
        Protocol protocol = Utils.retrieveProxy(this.urlParser, this.globalInfo);
        MariaDbConnection connection = new MariaDbConnection(protocol);
        MariaDbPooledConnection pooledConnection = this.createPoolConnection(connection);
        if (this.options.staticGlobal) {
            if (this.globalInfo == null) {
                this.initializePoolGlobalState(connection);
            }
            connection.setDefaultTransactionIsolation(this.globalInfo.getDefaultTransactionIsolation());
        } else {
            connection.setDefaultTransactionIsolation(connection.getTransactionIsolation());
        }
        if (this.poolState.get() == 0 && this.totalConnection.incrementAndGet() <= this.options.maxPoolSize) {
            this.idleConnections.addFirst(pooledConnection);
            if (logger.isDebugEnabled()) {
                logger.debug("pool {} new physical connection {} created (total:{}, active:{}, pending:{})", this.poolTag, connection.getServerThreadId(), this.totalConnection.get(), this.getActiveConnections(), this.pendingRequestNumber.get());
            }
            return;
        }
        this.silentCloseConnection(pooledConnection);
    }

    private MariaDbPooledConnection getIdleConnection() throws InterruptedException {
        return this.getIdleConnection(0L, TimeUnit.NANOSECONDS);
    }

    private MariaDbPooledConnection getIdleConnection(long timeout, TimeUnit timeUnit) throws InterruptedException {
        while (true) {
            long connId;
            MariaDbPooledConnection item;
            block5: {
                MariaDbPooledConnection mariaDbPooledConnection = item = timeout == 0L ? this.idleConnections.pollFirst() : this.idleConnections.pollFirst(timeout, timeUnit);
                if (item == null) break;
                MariaDbConnection connection = item.getConnection();
                connId = connection.getServerThreadId();
                try {
                    if (TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - item.getLastUsed().get()) > (long)this.options.poolValidMinDelay) {
                        if (connection.isValid(10)) {
                            item.lastUsedToNow();
                            return item;
                        }
                        break block5;
                    }
                    item.lastUsedToNow();
                    return item;
                }
                catch (SQLException sQLException) {
                    // empty catch block
                }
            }
            this.silentAbortConnection(item);
            this.addConnectionRequest();
            if (!logger.isDebugEnabled()) continue;
            logger.debug("pool {} connection {} removed from pool due to failed validation (total:{}, active:{}, pending:{})", this.poolTag, connId, this.totalConnection.get(), this.getActiveConnections(), this.pendingRequestNumber.get());
        }
        return null;
    }

    private void silentCloseConnection(MariaDbPooledConnection item) {
        try {
            item.close();
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
    }

    private void silentAbortConnection(MariaDbPooledConnection item) {
        try {
            item.abort(this.poolExecutor);
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
    }

    private MariaDbPooledConnection createPoolConnection(MariaDbConnection connection) {
        MariaDbPooledConnection pooledConnection = new MariaDbPooledConnection(connection);
        pooledConnection.addConnectionEventListener(new ConnectionEventListener(){

            @Override
            public void connectionClosed(ConnectionEvent event) {
                MariaDbPooledConnection item = (MariaDbPooledConnection)event.getSource();
                if (Pool.this.poolState.get() == 0) {
                    try {
                        if (!Pool.this.idleConnections.contains(item)) {
                            item.getConnection().pooledConnection = null;
                            item.getConnection().reset();
                            item.getConnection().pooledConnection = item;
                            Pool.this.idleConnections.addFirst(item);
                        }
                    }
                    catch (SQLException sqle) {
                        Pool.this.totalConnection.decrementAndGet();
                        Pool.this.silentCloseConnection(item);
                        logger.debug("connection {} removed from pool {} due to error during reset (total:{}, active:{}, pending:{})", item.getConnection().getServerThreadId(), Pool.this.poolTag, Pool.this.totalConnection.get(), Pool.this.getActiveConnections(), Pool.this.pendingRequestNumber.get());
                    }
                } else {
                    try {
                        item.close();
                    }
                    catch (SQLException sQLException) {
                        // empty catch block
                    }
                    Pool.this.totalConnection.decrementAndGet();
                }
            }

            @Override
            public void connectionErrorOccurred(ConnectionEvent event) {
                MariaDbPooledConnection item = (MariaDbPooledConnection)event.getSource();
                Pool.this.totalConnection.decrementAndGet();
                Pool.this.idleConnections.remove(item);
                Pool.this.silentCloseConnection(item);
                Pool.this.idleConnections.stream().forEach(c -> c.ensureValidation());
                Pool.this.addConnectionRequest();
                logger.debug("connection {} removed from pool {} due to having throw a Connection exception (total:{}, active:{}, pending:{})", item.getConnection().getServerThreadId(), Pool.this.poolTag, Pool.this.totalConnection.get(), Pool.this.getActiveConnections(), Pool.this.pendingRequestNumber.get());
            }
        });
        return pooledConnection;
    }

    public MariaDbConnection getConnection() throws SQLException {
        this.pendingRequestNumber.incrementAndGet();
        try {
            MariaDbPooledConnection pooledConnection = this.getIdleConnection(this.totalConnection.get() > 4 ? 0L : 50L, TimeUnit.MICROSECONDS);
            if (pooledConnection != null) {
                MariaDbConnection mariaDbConnection = pooledConnection.getConnection();
                return mariaDbConnection;
            }
            this.addConnectionRequest();
            pooledConnection = this.getIdleConnection(TimeUnit.MILLISECONDS.toNanos(this.options.connectTimeout), TimeUnit.NANOSECONDS);
            if (pooledConnection != null) {
                MariaDbConnection mariaDbConnection = pooledConnection.getConnection();
                return mariaDbConnection;
            }
            try {
                throw ExceptionFactory.INSTANCE.create(String.format("No connection available within the specified time (option 'connectTimeout': %s ms)", NumberFormat.getInstance().format(this.options.connectTimeout)));
            }
            catch (InterruptedException interrupted) {
                throw ExceptionFactory.INSTANCE.create("Thread was interrupted", "70100", interrupted);
            }
        }
        finally {
            this.pendingRequestNumber.decrementAndGet();
        }
    }

    public MariaDbConnection getConnection(String username, String password) throws SQLException {
        try {
            if ((this.urlParser.getUsername() != null ? this.urlParser.getUsername().equals(username) : username == null) && (this.urlParser.getPassword() != null ? this.urlParser.getPassword().equals(password) : password == null)) {
                return this.getConnection();
            }
            UrlParser tmpUrlParser = (UrlParser)this.urlParser.clone();
            tmpUrlParser.setUsername(username);
            tmpUrlParser.setPassword(password);
            Protocol protocol = Utils.retrieveProxy(tmpUrlParser, this.globalInfo);
            return new MariaDbConnection(protocol);
        }
        catch (CloneNotSupportedException cloneException) {
            throw new SQLException("Error getting connection, parameters cannot be cloned", cloneException);
        }
    }

    private String generatePoolTag(int poolIndex) {
        if (this.options.poolName == null) {
            this.options.poolName = "MariaDB-pool";
        }
        return this.options.poolName + "-" + poolIndex;
    }

    public UrlParser getUrlParser() {
        return this.urlParser;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws InterruptedException {
        Pool pool = this;
        synchronized (pool) {
            Pools.remove(this);
            this.poolState.set(1);
            this.pendingRequestNumber.set(0);
            this.scheduledFuture.cancel(false);
            this.connectionAppender.shutdown();
            try {
                this.connectionAppender.awaitTermination(10L, TimeUnit.SECONDS);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            if (logger.isInfoEnabled()) {
                logger.info("closing pool {} (total:{}, active:{}, pending:{})", this.poolTag, this.totalConnection.get(), this.getActiveConnections(), this.pendingRequestNumber.get());
            }
            ThreadPoolExecutor connectionRemover = new ThreadPoolExecutor(this.totalConnection.get(), this.options.maxPoolSize, 10L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(this.options.maxPoolSize), new MariaDbThreadFactory(this.poolTag + "-destroyer"));
            long start = System.nanoTime();
            do {
                this.closeAll(connectionRemover, this.idleConnections);
                if (this.totalConnection.get() <= 0) continue;
                Thread.sleep(0L, 1000);
            } while (this.totalConnection.get() > 0 && TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - start) < 10L);
            if (this.totalConnection.get() > 0 || this.idleConnections.isEmpty()) {
                this.closeAll(connectionRemover, this.idleConnections);
            }
            connectionRemover.shutdown();
            try {
                this.unRegisterJmx();
            }
            catch (Exception exception) {
                // empty catch block
            }
            connectionRemover.awaitTermination(10L, TimeUnit.SECONDS);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void closeAll(ExecutorService connectionRemover, Collection<MariaDbPooledConnection> collection) {
        Collection<MariaDbPooledConnection> collection2 = collection;
        synchronized (collection2) {
            for (MariaDbPooledConnection item : collection) {
                collection.remove(item);
                this.totalConnection.decrementAndGet();
                try {
                    item.abort(connectionRemover);
                }
                catch (SQLException sQLException) {}
            }
        }
    }

    private void initializePoolGlobalState(MariaDbConnection connection) throws SQLException {
        try (Statement stmt = connection.createStatement();){
            int major;
            String sql = "SELECT @@max_allowed_packet,@@wait_timeout,@@autocommit,@@auto_increment_increment,@@time_zone,@@system_time_zone,@@tx_isolation";
            if (!connection.isServerMariaDb() && ((major = connection.getMetaData().getDatabaseMajorVersion()) >= 8 && connection.versionGreaterOrEqual(8, 0, 3) || major < 8 && connection.versionGreaterOrEqual(5, 7, 20))) {
                sql = "SELECT @@max_allowed_packet,@@wait_timeout,@@autocommit,@@auto_increment_increment,@@time_zone,@@system_time_zone,@@transaction_isolation";
            }
            try (ResultSet rs = stmt.executeQuery(sql);){
                rs.next();
                int transactionIsolation = Utils.transactionFromString(rs.getString(7));
                this.globalInfo = new GlobalStateInfo(rs.getLong(1), rs.getInt(2), rs.getBoolean(3), rs.getInt(4), rs.getString(5), rs.getString(6), transactionIsolation);
                this.maxIdleTime = Math.min(this.options.maxIdleTime, this.globalInfo.getWaitTimeout() - 45);
            }
        }
    }

    public String getPoolTag() {
        return this.poolTag;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || this.getClass() != obj.getClass()) {
            return false;
        }
        Pool pool = (Pool)obj;
        return this.poolTag.equals(pool.poolTag);
    }

    public int hashCode() {
        return this.poolTag.hashCode();
    }

    public GlobalStateInfo getGlobalInfo() {
        return this.globalInfo;
    }

    @Override
    public long getActiveConnections() {
        return this.totalConnection.get() - this.idleConnections.size();
    }

    @Override
    public long getTotalConnections() {
        return this.totalConnection.get();
    }

    @Override
    public long getIdleConnections() {
        return this.idleConnections.size();
    }

    @Override
    public long getConnectionRequests() {
        return this.pendingRequestNumber.get();
    }

    private void registerJmx() throws Exception {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        String jmxName = this.poolTag.replace(":", "_");
        ObjectName name = new ObjectName("org.mariadb.jdbc.pool:type=" + jmxName);
        if (!mbs.isRegistered(name)) {
            mbs.registerMBean(this, name);
        }
    }

    private void unRegisterJmx() throws Exception {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        String jmxName = this.poolTag.replace(":", "_");
        ObjectName name = new ObjectName("org.mariadb.jdbc.pool:type=" + jmxName);
        if (mbs.isRegistered(name)) {
            mbs.unregisterMBean(name);
        }
    }

    public List<Long> testGetConnectionIdleThreadIds() {
        ArrayList<Long> threadIds = new ArrayList<Long>();
        for (MariaDbPooledConnection pooledConnection : this.idleConnections) {
            threadIds.add(pooledConnection.getConnection().getServerThreadId());
        }
        return threadIds;
    }

    @Override
    public void resetStaticGlobal() {
        this.globalInfo = null;
    }
}

