/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.transaction.log;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.zip.Checksum;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.EnumSource;
import org.neo4j.io.fs.ChecksumMismatchException;
import org.neo4j.io.fs.ChecksumWriter;
import org.neo4j.io.fs.EphemeralFileSystemAbstraction;
import org.neo4j.io.fs.ReadAheadChannel;
import org.neo4j.io.fs.ReadPastEndException;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.memory.ByteBuffers;
import org.neo4j.test.extension.EphemeralFileSystemExtension;
import org.neo4j.test.extension.Inject;

@ExtendWith(value={EphemeralFileSystemExtension.class})
class ReadAheadChannelTest {
    @Inject
    protected EphemeralFileSystemAbstraction fileSystem;

    ReadAheadChannelTest() {
    }

    @ParameterizedTest
    @EnumSource(value=Constructors.class)
    void shouldThrowExceptionForReadAfterEOFIfNotEnoughBytesExist(Constructor constructor) throws Exception {
        File bytesReadTestFile = new File("bytesReadTest.txt");
        StoreChannel storeChannel = this.fileSystem.write(bytesReadTestFile);
        ByteBuffer buffer = ByteBuffers.allocate((int)1);
        buffer.put((byte)1);
        buffer.flip();
        storeChannel.writeAll(buffer);
        storeChannel.force(false);
        storeChannel.close();
        storeChannel = this.fileSystem.read(bytesReadTestFile);
        HookedReadAheadChannel channel = constructor.apply(storeChannel, ReadAheadChannel.DEFAULT_READ_AHEAD_SIZE);
        Assertions.assertEquals((byte)1, (byte)channel.get());
        Assertions.assertThrows(ReadPastEndException.class, () -> ((ReadAheadChannel)channel).get());
        Assertions.assertThrows(ReadPastEndException.class, () -> ((ReadAheadChannel)channel).get());
    }

    @ParameterizedTest
    @EnumSource(value=Constructors.class)
    void shouldReturnValueIfSufficientBytesAreBufferedEvenIfEOFHasBeenEncountered(Constructor constructor) throws Exception {
        File shortReadTestFile = new File("shortReadTest.txt");
        StoreChannel storeChannel = this.fileSystem.write(shortReadTestFile);
        ByteBuffer buffer = ByteBuffers.allocate((int)1);
        buffer.put((byte)1);
        buffer.flip();
        storeChannel.writeAll(buffer);
        storeChannel.force(false);
        storeChannel.close();
        storeChannel = this.fileSystem.read(shortReadTestFile);
        HookedReadAheadChannel channel = constructor.apply(storeChannel, ReadAheadChannel.DEFAULT_READ_AHEAD_SIZE);
        Assertions.assertThrows(ReadPastEndException.class, () -> ((ReadAheadChannel)channel).getShort());
        Assertions.assertEquals((byte)1, (byte)channel.get());
        Assertions.assertThrows(ReadPastEndException.class, () -> ((ReadAheadChannel)channel).get());
    }

    @ParameterizedTest
    @EnumSource(value=Constructors.class)
    void shouldHandleRunningOutOfBytesWhenRequestSpansMultipleFiles(Constructor constructor) throws Exception {
        StoreChannel storeChannel1 = this.fileSystem.write(new File("foo.1"));
        ByteBuffer buffer = ByteBuffers.allocate((int)2);
        buffer.put((byte)0);
        buffer.put((byte)0);
        buffer.flip();
        storeChannel1.writeAll(buffer);
        storeChannel1.force(false);
        storeChannel1.close();
        buffer.flip();
        StoreChannel storeChannel2 = this.fileSystem.read(new File("foo.2"));
        buffer.put((byte)0);
        buffer.put((byte)1);
        buffer.flip();
        storeChannel2.writeAll(buffer);
        storeChannel2.force(false);
        storeChannel2.close();
        storeChannel1 = this.fileSystem.read(new File("foo.1"));
        StoreChannel storeChannel2Copy = this.fileSystem.read(new File("foo.2"));
        HookedReadAheadChannel channel = constructor.apply(storeChannel1, ReadAheadChannel.DEFAULT_READ_AHEAD_SIZE);
        channel.nextChannelHook = storeChannel2Copy;
        Assertions.assertThrows(ReadPastEndException.class, () -> ((HookedReadAheadChannel)channel).getLong());
        Assertions.assertEquals((int)1, (int)channel.getInt());
        Assertions.assertThrows(ReadPastEndException.class, () -> ((HookedReadAheadChannel)channel).get());
    }

    @ParameterizedTest
    @EnumSource(value=Constructors.class)
    void shouldReturnPositionWithinBufferedStream(Constructor constructor) throws Exception {
        File file = new File("foo.txt");
        int readAheadSize = 512;
        int fileSize = readAheadSize * 8;
        this.createFile(this.fileSystem, file, fileSize);
        HookedReadAheadChannel bufferedReader = constructor.apply(this.fileSystem.read(file), readAheadSize);
        for (int i = 0; i < fileSize / 8; ++i) {
            Assertions.assertEquals((long)(8 * i), (long)bufferedReader.position());
            bufferedReader.getLong();
        }
        Assertions.assertEquals((long)fileSize, (long)bufferedReader.position());
        Assertions.assertThrows(ReadPastEndException.class, () -> ((ReadAheadChannel)bufferedReader).getLong());
        Assertions.assertEquals((long)fileSize, (long)bufferedReader.position());
    }

    @ParameterizedTest
    @EnumSource(value=Constructors.class)
    void validateChecksumOverStream(Constructor constructor) throws Exception {
        int checksumValue;
        Checksum checksum = (Checksum)ChecksumWriter.CHECKSUM_FACTORY.get();
        File file = new File("foo.1");
        try (StoreChannel storeChannel = this.fileSystem.write(file);){
            ByteBuffer buffer = ByteBuffers.allocate((int)6);
            buffer.put((byte)1);
            checksum.update(1);
            buffer.put((byte)2);
            checksum.update(2);
            checksumValue = (int)checksum.getValue();
            buffer.putInt(checksumValue);
            buffer.flip();
            storeChannel.writeAll(buffer);
            storeChannel.force(false);
        }
        HookedReadAheadChannel bufferedReader = constructor.apply(this.fileSystem.read(file), ReadAheadChannel.DEFAULT_READ_AHEAD_SIZE);
        Assertions.assertEquals((int)1, (int)bufferedReader.get());
        Assertions.assertEquals((int)2, (int)bufferedReader.get());
        Assertions.assertEquals((int)checksumValue, (int)bufferedReader.endChecksumAndValidate());
        Assertions.assertEquals((long)6L, (long)bufferedReader.position());
    }

    @ParameterizedTest
    @EnumSource(value=Constructors.class)
    void throwOnInvalidChecksum(Constructor constructor) throws Exception {
        Checksum checksum = (Checksum)ChecksumWriter.CHECKSUM_FACTORY.get();
        File file = new File("foo.1");
        try (StoreChannel storeChannel = this.fileSystem.write(file);){
            ByteBuffer buffer = ByteBuffers.allocate((int)6);
            buffer.put((byte)1);
            checksum.update(1);
            buffer.put((byte)2);
            checksum.update(2);
            int notChecksumValue = (int)checksum.getValue() + 1;
            buffer.putInt(notChecksumValue);
            buffer.flip();
            storeChannel.writeAll(buffer);
            storeChannel.force(false);
        }
        HookedReadAheadChannel bufferedReader = constructor.apply(this.fileSystem.read(file), ReadAheadChannel.DEFAULT_READ_AHEAD_SIZE);
        Assertions.assertEquals((int)1, (int)bufferedReader.get());
        Assertions.assertEquals((int)2, (int)bufferedReader.get());
        Assertions.assertThrows(ChecksumMismatchException.class, () -> ((ReadAheadChannel)bufferedReader).endChecksumAndValidate());
    }

    @ParameterizedTest
    @EnumSource(value=Constructors.class)
    void checksumIsCalculatedCorrectlyOverBuffersLargerThanReadAheadSize(Constructor constructor) throws Exception {
        int checksumValue;
        int i;
        Checksum checksum = (Checksum)ChecksumWriter.CHECKSUM_FACTORY.get();
        File file = new File("foo.1");
        int testSize = 100;
        try (StoreChannel storeChannel = this.fileSystem.write(file);){
            ByteBuffer buffer = ByteBuffers.allocate((int)(testSize + 4));
            for (i = 0; i < testSize; ++i) {
                buffer.put((byte)i);
                checksum.update(i);
            }
            checksumValue = (int)checksum.getValue();
            buffer.putInt(checksumValue);
            buffer.flip();
            storeChannel.writeAll(buffer);
            storeChannel.force(false);
        }
        HookedReadAheadChannel bufferedReader = constructor.apply(this.fileSystem.read(file), testSize / 2);
        byte[] in = new byte[testSize];
        bufferedReader.get(in, testSize);
        for (i = 0; i < testSize; ++i) {
            Assertions.assertEquals((int)i, (int)in[i]);
        }
        Assertions.assertEquals((int)checksumValue, (int)bufferedReader.endChecksumAndValidate());
    }

    private void createFile(EphemeralFileSystemAbstraction fsa, File name, int bufferSize) throws IOException {
        StoreChannel storeChannel = fsa.write(name);
        ByteBuffer buffer = ByteBuffers.allocate((int)bufferSize);
        for (int i = 0; i < bufferSize; ++i) {
            buffer.put((byte)i);
        }
        buffer.flip();
        storeChannel.writeAll(buffer);
        storeChannel.close();
    }

    static enum Constructors implements Constructor
    {
        HEAP_BUFFER{

            @Override
            public HookedReadAheadChannel apply(StoreChannel channel, int readAheadSize) {
                return new HookedReadAheadChannel(channel, ByteBuffers.allocate((int)readAheadSize));
            }
        }
        ,
        DIRECT_BUFFER{

            @Override
            public HookedReadAheadChannel apply(StoreChannel channel, int readAheadSize) {
                return new HookedReadAheadChannel(channel, ByteBuffers.allocateDirect((int)readAheadSize));
            }
        }
        ,
        INNER_BUFFER{

            @Override
            public HookedReadAheadChannel apply(StoreChannel channel, int readAheadSize) {
                return new HookedReadAheadChannel(channel, readAheadSize);
            }
        };

    }

    static interface Constructor {
        public HookedReadAheadChannel apply(StoreChannel var1, int var2);
    }

    private static class HookedReadAheadChannel
    extends ReadAheadChannel<StoreChannel> {
        StoreChannel nextChannelHook;

        HookedReadAheadChannel(StoreChannel channel, int readAheadSize) {
            super(channel, readAheadSize);
        }

        HookedReadAheadChannel(StoreChannel channel, ByteBuffer byteBuffer) {
            super(channel, byteBuffer);
        }

        protected StoreChannel next(StoreChannel channel) throws IOException {
            if (this.nextChannelHook != null) {
                StoreChannel next = this.nextChannelHook;
                this.nextChannelHook = null;
                return next;
            }
            return super.next(channel);
        }
    }
}

