// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.collection;

import org.refcodes.structure.Relation;

/**
 * Basic functionality being provided by any {@link Dictionary} (map) style
 * class.
 *
 * @param <K> The type of the key.
 * @param <V> The type of the value which relates to a key.
 */
public interface Dictionary<K, V> extends Keys<K, V>, Containable {

	/**
	 * Extends the {@link Dictionary} interface with {@link ClearableKeys}
	 * functionality (as of {@link #clear()}). The case of having a plain
	 * {@link Keys} without dedicated {@link MutableKeys#remove(Object)} or
	 * {@link MutableKeys#put(Object, Object)} methods but with a dedicated
	 * {@link #clear()} method seems to be quite common, therefore this
	 * interface has been provided.
	 * 
	 * @param <K> The type of the key.
	 * @param <V> The type of the value which relates to a key.
	 */
	public interface ClearableDictionary<K, V> extends Dictionary<K, V>, ClearableKeys<K, V> {}

	/**
	 * Extends the {@link Dictionary} with mutable (writable) functionality,
	 * especially by providing {@link #add(Object)} and {@link #remove(Object)}
	 * methods.
	 * 
	 * @param <K> The type of the key.
	 * @param <V> The type of the value which relates to a key.
	 */
	public interface MutableDictionary<K, V> extends ClearableDictionary<K, V>, MutableKeys<K, V> {

		/**
		 * Adds the given element related to the given key.
		 * 
		 * @param aKey The key for which to add the element.
		 * 
		 * @param aValue The value to be related with the given key.
		 * 
		 * @return The value being replaced by the provided value or null if
		 *         none value has been replaced.
		 */
		V put( K aKey, V aValue );

		/**
		 * Adds the given element related to the given key.
		 * 
		 * @param aKey The key for which to add the element.
		 * 
		 * @param aValue The value to be related with the given key.
		 * 
		 * @return The value being replaced by the provided value or null if
		 *         none value has been replaced.
		 */
		V put( Relation<K, V> aRelation );

		/**
		 * Provides a builder method for a {@link Relation} property returning
		 * the builder for applying multiple build operations.
		 * 
		 * @param <B> The builder to return in order to be able to apply
		 *        multiple build operations. * @param <K> The type of the key.
		 * @param <V> The type of the value which relates to a key.
		 */
		public interface DictionaryBuilder<K, V, T extends Relation<K, V>, B extends DictionaryBuilder<K, V, T, B>> {

			B withPut( K aKey, V aValue );

			B withPut( T aRelation );

		}
	}
}
