// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.component;

/**
 * This mixin might be implemented by a component in order to provide cease
 * facilities. Cease is the zombie between stop and destroy. A processed
 * sequence may be ceased instead of being stopped. Taken an animation; an
 * animation sequence is stopped so it stands still ("freezes") to be resumed
 * again (e.g. via a "start()" method). It may also get ceased instead of being
 * stopped, in that case it may get faded out or (if it is a sprite) it may
 * explode - so to gracefully finish off that animation sequence.
 */
public interface Ceasable {

	/**
	 * Ceases the component.
	 * 
	 * @throws CeaseException in case ceasing fails.
	 */
	void cease() throws CeaseException;

	/**
	 * The {@link CeaseAutomaton} interface defines those methods related to the
	 * cease life-cycle.
	 */
	public interface CeaseAutomaton extends Ceasable {

		/**
		 * Determines whether the component may get ceased.
		 * 
		 * @return True if {@link #cease()} is possible.
		 */
		boolean isCeasable();

		/**
		 * Determines whether the component is ceased.
		 * 
		 * @return True in case of being ceased, else false.
		 */
		boolean isCeased();
	}

	/**
	 * To enable the {@link Ceasable} functionality to be invoked in a builder
	 * chain.
	 *
	 * @param <B> The instance to be returned on which to apply succeeding builder
	 *        operations.
	 */
	public interface CeaseBuilder<B extends CeaseBuilder<B>> {

		/**
		 * Builder method for the {@link Cease#cease()} method.
		 * 
		 * @return The instance to be returned on which to apply succeeding builder
		 *         operations.
		 * 
		 * @throws CeaseException Thrown in case ceasing fails.
		 */
		B withCease() throws CeaseException;
	}

	/**
	 * See {@link Ceasable} without any checked exception being declared.
	 */
	public interface UncheckedCeasable extends Ceasable {

		/**
		 * See {@link Ceasable#cease()} without any checked exception being
		 * declared.
		 */
		void cease();
	}
}
