// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.component;

/**
 * This mixin might be implemented by a component in order to provide initialize
 * facilities wit a configuration provided via {@link #initialize(Object)}.
 * <p>
 * In case a no context is to be provided to the {@link #initialize(Object)}
 * method (as it may have been passed via the constructor), you may use the
 * {@link Initializable} interface with its {@link Initializable#initialize()}
 * method, which does not require any arguments specifying a context.
 *
 * @param <CTX> the context used to initialize the implementing instance.
 */
public interface Configurable<CTX> {

	/**
	 * Initializes and configures the component.
	 * 
	 * @param <CTX> the context used to initialize the implementing instance.
	 * 
	 * @param aContext The context to be passed to the implementing instance.
	 * 
	 * @throws ConfigureException Thrown in case initializing fails.
	 */
	void initialize( CTX aContext ) throws ConfigureException;

	/**
	 * The {@link ConfigureAutomaton} interface defines those methods related to
	 * the initialize life-cycle with a provided context.
	 * 
	 * @param <CTX> the context used to initialize the implementing instance.
	 */
	public interface ConfigureAutomaton<CTX> extends Configurable<CTX>, InitializedAccessor {

		/**
		 * Determines whether the component may get initialized.
		 * 
		 * @param <CTX> the context used to initialize the implementing
		 *        instance.
		 * 
		 * @param aContext The context to be passed to the implementing
		 *        instance.
		 * 
		 * @return True if {@link #initialize()} is possible.
		 */
		boolean isInitalizable( CTX aContext );
	}

	/**
	 * To enable the {@link Initializable} functionality to be invoked in a
	 * builder chain.
	 *
	 * @param <CTX> the context used to initialize the implementing instance.
	 * 
	 * @param <B> The instance to be returned on which to apply succeeding
	 *        builder operations.
	 */
	public interface ConfigureBuilder<CTX, B extends ConfigureBuilder<CTX, B>> {

		/**
		 * Builder method for the {@link Configurable#initialize(Object)}
		 * method.
		 * 
		 * @param <CTX> the context used to initialize the implementing
		 *        instance.
		 * 
		 * @param aContext The context to be passed to the implementing
		 *        instance.
		 * 
		 * @return The instance to be returned on which to apply succeeding
		 *         builder operations.
		 * 
		 * @throws ConfigureException Thrown in case initializing fails.
		 */
		B withInitialize( CTX aContext ) throws ConfigureException;
	}

}
