package org.refcodes.component;

import org.refcodes.component.ConnectionOpenable.ConnectionOpenAutomaton;

/**
 * The {@link ConnectionOpenableHandle} interface defines those methods related
 * to the handle based open/connect life-cycle.
 * <p>
 * The handle reference requires the {@link ConnectionOpenable} interface to be
 * implemented.
 *
 * @param <H> The type of the handle.
 *
 * @param <CON> The connection used to open the referenced instance.
 */
public interface ConnectionOpenableHandle<H, CON> {

	/**
	 * Determines whether the handle reference is {@link ConnectionOpenable} by
	 * implementing the {@link ConnectionOpenable} interface.
	 * 
	 * @param aHandle The handle to test whether the reference provides the
	 *        according functionality.
	 * 
	 * @return True in case the reference provides the according functionality.
	 * 
	 * @throws UnknownHandleRuntimeException in case the handle is unknown.
	 */
	boolean hasConnectionOpenable( H aHandle ) throws UnknownHandleRuntimeException;

	/**
	 * Open/connect the component identified by the given handle.
	 * 
	 * @param aConnection The connection to be passed to the implementing
	 *        instance.
	 * 
	 * @param aHandle The handle identifying the component.
	 * 
	 * @throws OpenException in case opening/connecting fails.
	 * 
	 * @throws UnsupportedHandleOperationRuntimeException in case the reference
	 *         of the handle does not support the requested operation.
	 * 
	 * @throws UnknownHandleRuntimeException in case the given handle is
	 *         unknown.
	 * 
	 * @throws IllegaleHandleStateChangeRuntimeException Thrown in case a state
	 *         change is not possible due to the current state the referenced
	 *         component is in.
	 */
	void open( H aHandle, CON aConnection ) throws OpenException, UnknownHandleRuntimeException, UnsupportedHandleOperationRuntimeException, IllegaleHandleStateChangeRuntimeException;

	/**
	 * The {@link ConnectionOpenAutomatonHandle} interface defines those methods
	 * related to the handle based open/connect life-cycle.
	 * 
	 * The handle reference requires the {@link ConnectionOpenAutomaton}
	 * interface to be implemented.
	 * 
	 * @param <H> The type of the handle.
	 * 
	 * @param <CON> the connection used to initialize the referenced instance.
	 */
	public interface ConnectionOpenAutomatonHandle<H, CON> extends ConnectionOpenableHandle<H, CON>, OpenedHandle<H> {

		/**
		 * Determines whether the handle reference is configurable by
		 * implementing the {@link ConnectionOpenAutomaton} interface.
		 * 
		 * @param aHandle The handle to test whether the reference provides the
		 *        according functionality.
		 * 
		 * @return True in case the reference provides the according
		 *         functionality.
		 * 
		 * @throws UnknownHandleRuntimeException in case the handle is unknown.
		 */
		boolean hasConnectionOpenAutomaton( H aHandle ) throws UnknownHandleRuntimeException;

		/**
		 * Determines whether the component identified by the given handle may
		 * get opened/connected.
		 * 
		 * @param aHandle The handle identifying the component.
		 * 
		 * @param aConnection The connection to be passed to the implementing
		 *        instance.
		 * 
		 * @return True if {@link #open(Object, Object)} is possible.
		 * 
		 * @throws UnsupportedHandleOperationRuntimeException in case the
		 *         reference of the handle does not support the requested
		 *         operation.
		 * 
		 * @throws UnknownHandleRuntimeException in case the given handle is
		 *         unknown.
		 */
		boolean isOpenable( H aHandle, CON aConnection ) throws UnknownHandleRuntimeException, UnsupportedHandleOperationRuntimeException;
	}
}
