package org.refcodes.component;

import org.refcodes.component.Decomposeable.DecomposeAutomaton;

/**
 * The {@link DecomposeableHandle} interface defines those methods related to
 * the handle based decompose life-cycle.
 * <p>
 * The handle reference requires the {@link Decomposeable} interface to be
 * implemented.
 *
 * @param <H> The type of the handle.
 */
public interface DecomposeableHandle<H> {

	/**
	 * Determines whether the handle reference is decomposable by implementing
	 * the {@link Decomposeable} interface.
	 * 
	 * @param aHandle The handle to test whether the reference provides the
	 *        according functionality.
	 * 
	 * @return True in case the reference provides the according functionality.
	 * 
	 * @throws UnknownHandleRuntimeException in case the handle is unknown.
	 */
	boolean hasDecomposeable( H aHandle ) throws UnknownHandleRuntimeException;

	/**
	 * Decomposes the component identified by the given handle.
	 * 
	 * @param aHandle The handle identifying the component.
	 * 
	 * @throws UnsupportedHandleOperationRuntimeException in case the reference
	 *         of the handle does not support the requested operation.
	 * @throws UnknownHandleRuntimeException in case the given handle is
	 *         unknown.
	 */
	void decompose( H aHandle ) throws UnknownHandleRuntimeException, UnsupportedHandleOperationRuntimeException;

	/**
	 * The {@link DecomposeAutomatonHandle} interface defines those methods
	 * related to the handle based decompose life-cycle.
	 * 
	 * The handle reference requires the {@link DecomposeAutomaton} interface to
	 * be implemented.
	 * 
	 * @param <H> The type of the handle.
	 */
	public interface DecomposeAutomatonHandle<H> extends DecomposeableHandle<H> {

		/**
		 * Determines whether the handle reference is decomposable by
		 * implementing the {@link DecomposeAutomaton} interface.
		 * 
		 * @param aHandle The handle to test whether the reference provides the
		 *        according functionality.
		 * 
		 * @return True in case the reference provides the according
		 *         functionality.
		 * 
		 * @throws UnknownHandleRuntimeException in case the handle is unknown.
		 */
		boolean hasDecomposeAutomaton( H aHandle ) throws UnknownHandleRuntimeException;

		/**
		 * Determines whether the component identified by the given handle may
		 * get decomposed.
		 * 
		 * @param aHandle The handle identifying the component.
		 * 
		 * @return True if {@link #destroy(Object)} is possible.
		 * 
		 * @throws UnsupportedHandleOperationRuntimeException in case the
		 *         reference of the handle does not support the requested
		 *         operation.
		 * @throws UnknownHandleRuntimeException in case the given handle is
		 *         unknown.
		 */
		boolean isDecomposeable( H aHandle ) throws UnknownHandleRuntimeException, UnsupportedHandleOperationRuntimeException;

		/**
		 * Decomposes the component identified by the given handle.
		 * 
		 * @param aHandle The handle identifying the component.
		 * 
		 * @throws UnsupportedHandleOperationRuntimeException in case the
		 *         reference of the handle does not support the requested
		 *         operation.
		 * @throws UnknownHandleRuntimeException in case the given handle is
		 *         unknown.
		 */
		@Override
		void decompose( H aHandle ) throws UnknownHandleRuntimeException, UnsupportedHandleOperationRuntimeException;
	}
}
