package org.refcodes.component;

import org.refcodes.mixin.Disposable.DisposeSupport;

/**
 * The {@link DisposableHandle} interface defines those methods related to the
 * handle based dispose life-cycle.
 * <p>
 * The handle reference requires the {@link DisposeSupport} interface to be
 * implemented.
 *
 * @param <H> The type of the handle.
 */
public interface DisposableHandle<H> {

	/**
	 * Determines whether the handle reference is disposable by implementing the
	 * {@link DisposeSupport} interface.
	 * 
	 * @param aHandle The handle to test whether the reference provides the
	 *        according functionality.
	 * 
	 * @return True in case the reference provides the according functionality.
	 * 
	 * @throws UnknownHandleRuntimeException in case the handle is unknown.
	 */
	boolean hasDisposable( H aHandle ) throws UnknownHandleRuntimeException;

	/**
	 * Disposes the component identified by the given handle.
	 * 
	 * @param aHandle The handle identifying the component.
	 * 
	 * @throws UnsupportedHandleOperationRuntimeException in case the reference
	 *         of the handle does not support the requested operation.
	 * 
	 * @throws UnknownHandleRuntimeException in case the given handle is
	 *         unknown.
	 * 
	 * @throws IllegaleHandleStateChangeRuntimeException Thrown in case a state
	 *         change is not possible due to the current state the referenced
	 *         component is in.
	 */
	void dispose( H aHandle ) throws UnknownHandleRuntimeException, UnsupportedHandleOperationRuntimeException, IllegaleHandleStateChangeRuntimeException;

	/**
	 * The {@link DisposeAutomatonHandle} interface defines those methods
	 * related to the handle based dispose life-cycle.
	 * 
	 * The handle reference requires the {@link DisposeAutomaton} interface to
	 * be implemented.
	 * 
	 * @param <H> The type of the handle.
	 */
	public interface DisposeAutomatonHandle<H> extends DisposableHandle<H> {

		/**
		 * Determines whether the handle reference is disposable by implementing
		 * the {@link DisposeAutomaton} interface.
		 * 
		 * @param aHandle The handle to test whether the reference provides the
		 *        according functionality.
		 * 
		 * @return True in case the reference provides the according
		 *         functionality.
		 * 
		 * @throws UnknownHandleRuntimeException in case the handle is unknown.
		 */
		boolean hasDisposeAutomaton( H aHandle ) throws UnknownHandleRuntimeException;

		/**
		 * Determines whether the component identified by the given handle may
		 * get disposed.
		 * 
		 * @param aHandle The handle identifying the component.
		 * 
		 * @return True if {@link #dispose(Object)} is possible.
		 * 
		 * @throws UnsupportedHandleOperationRuntimeException in case the
		 *         reference of the handle does not support the requested
		 *         operation.
		 * @throws UnknownHandleRuntimeException in case the given handle is
		 *         unknown.
		 */
		boolean isDisposable( H aHandle ) throws UnknownHandleRuntimeException, UnsupportedHandleOperationRuntimeException;

		/**
		 * Determines whether the component identified by the given handle is
		 * disposed.
		 * 
		 * @param aHandle The handle identifying the component.
		 * 
		 * @return True in case of being disposed, else false.
		 * 
		 * @throws UnsupportedHandleOperationRuntimeException in case the
		 *         reference of the handle does not support the requested
		 *         operation.
		 * @throws UnknownHandleRuntimeException in case the given handle is
		 *         unknown.
		 */
		boolean isDisposed( H aHandle ) throws UnknownHandleRuntimeException, UnsupportedHandleOperationRuntimeException;
	}
}
