// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.component;

import org.refcodes.component.LifeCycleComponentHandle.LifeCycleAutomatonHandle;

/**
 * A component implementing the {@link LifeCycleComponent} interface supports a
 * life-cycle. I.e. such a component may be instructed from the outside to run
 * through several stages from getting started till being destroyed. The valid
 * state changes are mainly as follows:
 * <p>
 * "initialize" - "start" - "pause" - "resume" - "stop" - "destroy"
 * <p>
 * <p>
 * For example: "initialize" - "start" - "pause" - "resume" - "pause" - "resume"
 * - "stop" - "start" - "pause" - "resume" - "stop" - "destroy"
 * <p>
 * The {@link LifeCycleAutomatonHandle} is a component managing various
 * {@link LifeCycleComponent}s each identified by a dedicated handle. Operations
 * on the {@link LifeCycleComponent} are invoked by this
 * {@link LifeCycleAutomatonHandle} with a handle identifying the according
 * {@link LifeCycleComponent}.
 * <p>
 * The {@link LifeCycleComponent} contains the business-logic where as the
 * {@link LifeCycleAutomatonHandle} provides the frame for managing this
 * business-logic. The {@link LifeCycleAutomaton} takes care of the correct
 * life-cycle applied on a {@link LifeCycleComponent}.
 * */
public interface LifeCycleComponent extends Initializable, Startable, Pausable, Resumable, Stoppable, Destroyable {

	/**
	 * A system implementing the {@link LifeCycleAutomaton} interface supports
	 * managing {@link LifeCycleComponent} instances and takes care that the
	 * life-cycle stages are invoked in the correct order by throwing according
	 * exceptions in case the life-cycle is invoked in the wrong order.
	 * 
	 * A {@link LifeCycleAutomaton} may be used to wrap a
	 * {@link LifeCycleComponent} by a {@link LifeCycleAutomatonHandle} for
	 * managing {@link LifeCycleComponent} instances.
	 * 
	 * The {@link LifeCycleComponent} contains the business-logic where as the
	 * {@link LifeCycleAutomatonHandle} provides the frame for managing this
	 * business-logic. The {@link LifeCycleAutomaton} takes care of the correct
	 * life-cycle applied on a {@link LifeCycleComponent}.
	 */
	public interface LifeCycleAutomaton extends LifeCycleComponent, InitializeAutomaton, StartAutomaton, PauseAutomaton, ResumeAutomaton, StopAutomaton, DestroyAutomaton, LifeCycleStatusAccessor {}

	
	/**
	 * Same as the {@link LifeCycleComponent} though without the need to
	 * try-catch any exceptions on the various life-cycle stages.
	 */
	public interface UncheckedLifeCycleComponent extends LifeCycleComponent, UncheckedInitializable, UncheckedStartable, UncheckedPausable, UncheckedResumable, UncheckedStoppable, Destroyable {}
	

}
