package org.refcodes.component;

import org.refcodes.component.ConnectionOpenable.ConnectionOpenAutomaton;

/**
 * The {@link OpenableHandle} interface defines those methods related to the
 * handle based open/connect life-cycle.
 * <p>
 * The handle reference requires the {@link Openable} interface to be
 * implemented.
 *
 * @param <H> The type of the handle.
 */
public interface OpenableHandle<H> {

	/**
	 * Determines whether the handle reference is openable by implementing the
	 * {@link Openable} interface.
	 * 
	 * @param aHandle The handle to test whether the reference provides the
	 *        according functionality.
	 * 
	 * @return True in case the reference provides the according functionality.
	 * 
	 * @throws UnknownHandleRuntimeException in case the handle is unknown.
	 */
	boolean hasOpenable( H aHandle ) throws UnknownHandleRuntimeException;

	/**
	 * Open/connect the component identified by the given handle.
	 * 
	 * @param aHandle The handle identifying the component.
	 * 
	 * @throws OpenException in case opening/connecting fails.
	 * 
	 * @throws UnsupportedHandleOperationRuntimeException in case the reference
	 *         of the handle does not support the requested operation.
	 * 
	 * @throws UnknownHandleRuntimeException in case the given handle is
	 *         unknown.
	 * 
	 * @throws IllegaleHandleStateChangeRuntimeException Thrown in case a state
	 *         change is not possible due to the current state the referenced
	 *         component is in.
	 */
	void open( H aHandle ) throws OpenException, UnknownHandleRuntimeException, UnsupportedHandleOperationRuntimeException, IllegaleHandleStateChangeRuntimeException;

	/**
	 * The {@link OpenAutomatonHandle} interface defines those methods related
	 * to the handle based open/connect life-cycle.
	 * 
	 * The handle reference requires the {@link ConnectionOpenAutomaton}
	 * interface to be implemented.
	 * 
	 * @param <H> The type of the handle.
	 */
	public interface OpenAutomatonHandle<H> extends OpenableHandle<H>, OpenedHandle<H> {

		/**
		 * Determines whether the handle reference is openable by implementing
		 * the {@link ConnectionOpenAutomaton} interface.
		 * 
		 * @param aHandle The handle to test whether the reference provides the
		 *        according functionality.
		 * 
		 * @return True in case the reference provides the according
		 *         functionality.
		 * 
		 * @throws UnknownHandleRuntimeException in case the handle is unknown.
		 */
		boolean hasOpenAutomaton( H aHandle ) throws UnknownHandleRuntimeException;

		/**
		 * Determines whether the component identified by the given handle may
		 * get opened/connected.
		 * 
		 * @param aHandle The handle identifying the component.
		 * 
		 * @return True if {@link #open(Object)} is possible.
		 * 
		 * @throws UnsupportedHandleOperationRuntimeException in case the
		 *         reference of the handle does not support the requested
		 *         operation.
		 * 
		 * @throws UnknownHandleRuntimeException in case the given handle is
		 *         unknown.
		 */
		boolean isOpenable( H aHandle ) throws UnknownHandleRuntimeException, UnsupportedHandleOperationRuntimeException;
	}
}
