package org.refcodes.component;

import org.refcodes.component.Stoppable.StopAutomaton;

/**
 * The {@link StoppableHandle} interface defines those methods related to the
 * handle based stop life-cycle.
 * <p>
 * The handle reference requires the {@link Stoppable} interface to be
 * implemented.
 *
 * @param <H> The type of the handle.
 */
public interface StoppableHandle<H> {

	/**
	 * Determines whether the handle reference is stoppable by implementing the
	 * {@link Stoppable} interface.
	 * 
	 * @param aHandle The handle to test whether the reference provides the
	 *        according functionality.
	 * 
	 * @return True in case the reference provides the according functionality.
	 * 
	 * @throws UnknownHandleRuntimeException in case the handle is unknown.
	 */
	boolean hasStoppable( H aHandle ) throws UnknownHandleRuntimeException;

	/**
	 * Stops the component identified by the given handle.
	 * 
	 * @param aHandle The handle identifying the component.
	 * 
	 * @throws StopException in case stopping fails.
	 * 
	 * @throws UnsupportedHandleOperationRuntimeException in case the reference
	 *         of the handle does not support the requested operation.
	 * 
	 * @throws UnknownHandleRuntimeException in case the given handle is
	 *         unknown.
	 * 
	 * @throws IllegaleHandleStateChangeRuntimeException Thrown in case a state
	 *         change is not possible due to the current state the referenced
	 *         component is in.
	 */
	void stop( H aHandle ) throws StopException, UnknownHandleRuntimeException, UnsupportedHandleOperationRuntimeException, IllegaleHandleStateChangeRuntimeException;

	/**
	 * The {@link StopAutomatonHandle} interface defines those methods related
	 * to the handle based stop life-cycle.
	 * 
	 * The handle reference requires the {@link StopAutomaton} interface to be
	 * implemented.
	 * 
	 * @param <H> The type of the handle.
	 */
	public interface StopAutomatonHandle<H> extends StoppableHandle<H> {

		/**
		 * Determines whether the handle reference is stoppable by implementing
		 * the {@link StopAutomaton} interface.
		 * 
		 * @param aHandle The handle to test whether the reference provides the
		 *        according functionality.
		 * 
		 * @return True in case the reference provides the according
		 *         functionality.
		 * 
		 * @throws UnknownHandleRuntimeException in case the handle is unknown.
		 */
		boolean hasStopAutomaton( H aHandle ) throws UnknownHandleRuntimeException;

		/**
		 * Determines whether the component identified by the given handle may
		 * get stopped.
		 * 
		 * @param aHandle The handle identifying the component.
		 * 
		 * @return True if {@link #stop(Object)} is possible.
		 * 
		 * @throws UnsupportedHandleOperationRuntimeException in case the
		 *         reference of the handle does not support the requested
		 *         operation.
		 * @throws UnknownHandleRuntimeException in case the given handle is
		 *         unknown.
		 */
		boolean isStoppable( H aHandle ) throws UnknownHandleRuntimeException, UnsupportedHandleOperationRuntimeException;

		/**
		 * Determines whether the component identified by the given handle is
		 * stopped.
		 * 
		 * @param aHandle The handle identifying the component.
		 * 
		 * @return True in case of being stopped, else false.
		 * 
		 * @throws UnsupportedHandleOperationRuntimeException in case the
		 *         reference of the handle does not support the requested
		 *         operation.
		 * @throws UnknownHandleRuntimeException in case the given handle is
		 *         unknown.
		 */
		boolean isStopped( H aHandle ) throws UnknownHandleRuntimeException, UnsupportedHandleOperationRuntimeException;
	}
}
