// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.component.impls;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.refcodes.component.Component;
import org.refcodes.component.ComponentUtility;
import org.refcodes.component.CompositeComponent;
import org.refcodes.component.ConfigureException;
import org.refcodes.component.InitializeException;
import org.refcodes.component.OpenException;
import org.refcodes.component.PauseException;
import org.refcodes.component.ResumeException;
import org.refcodes.component.StartException;
import org.refcodes.component.StopException;
import org.refcodes.controlflow.ExecutionStrategy;

/**
 * The {@link CompositeComponentImpl} is an implementation of the
 * {@link CompositeComponent}. To make sure that the state change requests you
 * require are supported by the managed {@link Component} instances, specify the
 * according type <C> as generic type argument.
 *
 * @param <C> The type of the {@link Component} supported by the
 *        {@link CompositeComponentImpl}.
 */
public class CompositeComponentImpl<C extends Component> implements CompositeComponent {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Set<C> _components;
	private ExecutionStrategy _strategy;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates a {@link CompositeComponentImpl} containing the provided
	 * components. The {@link ExecutionStrategy#JOIN} {@link ExecutionStrategy}
	 * is used by default.
	 * 
	 * @param aComponents The components to be managed by the
	 *        {@link CompositeComponentImpl}.
	 */
	public CompositeComponentImpl( Collection<C> aComponents ) {
		this( ExecutionStrategy.JOIN, aComponents );
	}

	/**
	 * Creates a {@link CompositeComponentImpl} containing the provided
	 * components.
	 * 
	 * @param aStrategy The {@link ExecutionStrategy} for executing the state
	 *        change requests.
	 * 
	 * @param aComponents The components to be managed by the
	 *        {@link CompositeComponentImpl}.
	 */
	public CompositeComponentImpl( ExecutionStrategy aStrategy, Collection<C> aComponents ) {
		_components = new HashSet<C>( aComponents );
		_strategy = aStrategy;
	}

	/**
	 * Creates a {@link CompositeComponentImpl} containing the provided
	 * components. The {@link ExecutionStrategy#JOIN} {@link ExecutionStrategy}
	 * is used by default.
	 * 
	 * @param aComponents The components to be managed by the
	 *        {@link CompositeComponentImpl}.
	 */
	@SafeVarargs
	public CompositeComponentImpl( C... aComponents ) {
		this( ExecutionStrategy.JOIN, aComponents );
	}

	/**
	 * Creates a {@link CompositeComponentImpl} containing the provided
	 * components.
	 * 
	 * @param aStrategy The {@link ExecutionStrategy} for executing the state
	 *        change requests.
	 * 
	 * @param aComponents The components to be managed by the
	 *        {@link CompositeComponentImpl}.
	 */
	@SafeVarargs
	public CompositeComponentImpl( ExecutionStrategy aStrategy, C... aComponents ) {
		List<C> theComponents = Arrays.asList( aComponents );
		_components = new HashSet<C>( theComponents );
		_strategy = aStrategy;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public void flush() throws OpenException {
		ComponentUtility.flush( _strategy, _components );
	}

	@Override
	public void decompose() {
		ComponentUtility.decompose( _strategy, _components );
	}

	@Override
	public void destroy() {
		ComponentUtility.destroy( _strategy, _components );
	}

	@Override
	public void stop() throws StopException {
		ComponentUtility.stop( _strategy, _components );
	}

	@Override
	public void resume() throws ResumeException {
		ComponentUtility.resume( _strategy, _components );
	}

	@Override
	public void pause() throws PauseException {
		ComponentUtility.pause( _strategy, _components );
	}

	@Override
	public void start() throws StartException {
		ComponentUtility.start( _strategy, _components );
	}

	@Override
	public void initialize() throws InitializeException {
		ComponentUtility.initialize( _strategy, _components );
	}

	@Override
	public void reset() {
		ComponentUtility.reset( _strategy, _components );
	}

	@Override
	public void open() throws OpenException {
		ComponentUtility.open( _strategy, _components );
	}

	@Override
	public void close() {
		ComponentUtility.close( _strategy, _components );
	}

	@Override
	public void dispose() {
		ComponentUtility.dispose( _strategy, _components );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Provides access to the {@link ExecutionStrategy}.
	 * 
	 * @return The {@link ExecutionStrategy} being set.
	 */
	protected ExecutionStrategy getExecutionStrategy() {
		return _strategy;
	}

	/**
	 * Provides access to the {@link Component} instances.
	 * 
	 * @return The {@link Component} instances being set.
	 */
	protected Set<C> getComponents() {
		return _components;
	}

	/**
	 * The {@link ExtendedCompositeComponentImpl} is an implementation of the
	 * {@link ExtendedCompositeComponent}. To make sure that the state change
	 * requests you require are supported by the managed {@link Component}
	 * instances, specify the according type <C> as generic type argument.
	 * 
	 * @param <C> The type of the {@link Component} supported by the
	 *        {@link ExtendedCompositeComponentImpl}.
	 */
	public static class ExtendedCompositeComponentImpl<C extends Component, CTX, CON> extends CompositeComponentImpl<C> implements ExtendedCompositeComponent<CTX, CON> {

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////


		@SafeVarargs
		public ExtendedCompositeComponentImpl( C... aComponents ) {
			super( aComponents );
		}


		public ExtendedCompositeComponentImpl( Collection<C> aComponents ) {
			super( aComponents );
		}


		@SafeVarargs
		public ExtendedCompositeComponentImpl( ExecutionStrategy aStrategy, C... aComponents ) {
			super( aStrategy, aComponents );
		}


		public ExtendedCompositeComponentImpl( ExecutionStrategy aStrategy, Collection<C> aComponents ) {
			super( aStrategy, aComponents );
		}

		// /////////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////////

		@Override
		public void initialize( CTX aContext ) throws ConfigureException {
			ComponentUtility.initialize( getExecutionStrategy(), aContext, getComponents() );
		}

		@Override
		public void open( CON aConnection ) throws OpenException {
			ComponentUtility.open( getExecutionStrategy(), aConnection, getComponents() );
		}
	}
}
