// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cryptography.tests;

import static org.junit.Assert.assertEquals;

import org.junit.Test;
import org.refcodes.cryptography.DecryptionProvider;
import org.refcodes.cryptography.EncryptionProvider;
import org.refcodes.cryptography.EncryptionServer;
import org.refcodes.cryptography.EncryptionService;
import org.refcodes.cryptography.impls.DecryptionProviderImpl;
import org.refcodes.cryptography.impls.EncryptionProviderImpl;
import org.refcodes.cryptography.impls.InMemoryDecryptionServerImpl;
import org.refcodes.cryptography.impls.InMemoryEncryptionServerImpl;
import org.refcodes.cryptography.impls.LoopbackDecryptionServiceImpl;
import org.refcodes.cryptography.impls.LoopbackEncryptionServiceImpl;
import org.refcodes.cryptography.utils.CryptographyUtility;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.factories.impls.RuntimeLoggerFactorySingleton;

@SuppressWarnings("deprecation")
public class CryptographyTest {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.getInstance().createInstance();

	private static final String HALLO_WELT = "Hallo Welt";

	private static final String NAMESPACE = "test";

	@Test
	public void testCryptographyInfrystructure() throws Exception {
		InMemoryDecryptionServerImpl theDecryptionServer = new InMemoryDecryptionServerImpl();
		LoopbackDecryptionServiceImpl theDecryptionService = new LoopbackDecryptionServiceImpl( NAMESPACE, theDecryptionServer );
		theDecryptionService.setCipherVersionsExpireTimeInMs( 500 );
		DecryptionProvider theDecryptionProvider = new DecryptionProviderImpl( theDecryptionService );
		EncryptionServer theEncryptionServer = new InMemoryEncryptionServerImpl( theDecryptionServer );
		EncryptionService theEncryptionService = new LoopbackEncryptionServiceImpl( NAMESPACE, theEncryptionServer );
		EncryptionProvider theEncryptionProvider = new EncryptionProviderImpl( theEncryptionService );

		// @formatter:off
		// CryptographyProvider theCryptographyProvider = new CryptographyProviderImpl( theEncryptionProvider, theDecryptionProvider );
		// @formatter:on

		String theText = "Hallo Welt!";
		String theEncryptedText = theEncryptionProvider.toEncrypted( theText );

		String theDecryptedText = theDecryptionProvider.toDecrypted( theEncryptedText );
		LOGGER.info( theEncryptedText + " -> " + theDecryptedText );
		assertEquals( theText, theDecryptedText );

		theEncryptionProvider.nextCipherVersion();
		String theEncryptedText2 = theEncryptionProvider.toEncrypted( theText );

		String theDecryptedText2 = theDecryptionProvider.toDecrypted( theEncryptedText );
		LOGGER.info( theEncryptedText2 + " -> " + theDecryptedText2 );
		assertEquals( theText, theDecryptedText2 );

		theDecryptedText = theDecryptionProvider.toDecrypted( theEncryptedText );
		LOGGER.info( theEncryptedText + " -> " + theDecryptedText );
		assertEquals( theText, theDecryptedText );

		String eEncryptedText, eDecryptedText;
		for ( int i = 0; i < 50; i++ ) {
			theEncryptionProvider.nextCipherVersion();
			eEncryptedText = theEncryptionProvider.toEncrypted( theText );
			eDecryptedText = theDecryptionProvider.toDecrypted( theEncryptedText );
			LOGGER.info( eEncryptedText + " -> " + eDecryptedText );
			assertEquals( theText, eDecryptedText );
		}
	}

	@Test
	public void testSymetricCryptography() throws Exception {

		// WARMUP:
		String theText = HALLO_WELT;
		String theEncryptedText = CryptographyUtility.toEncryptedText( theText );
		LOGGER.info( "\"" + theText + "\" --> \"" + theEncryptedText + "\"" );
		String theDecryptedText = CryptographyUtility.toDecryptedText( theEncryptedText );
		assertEquals( theText, theDecryptedText );
		assertEquals( theDecryptedText, CryptographyUtility.toDecryptedText( theEncryptedText ) );

		// TESTSTRING:
		String theTestString = "0123456789 0123456789 0123456789 0123456789 0123456789 0123456789 0123456789";

		// ENCRYPTION:
		long theStartTime = System.currentTimeMillis();
		int theRuns = 10000;
		for ( int i = 0; i < theRuns; i++ ) {
			theEncryptedText = CryptographyUtility.toEncryptedText( (theTestString + i) );
		}
		long theEndTime = System.currentTimeMillis();
		LOGGER.info( "SYMETRIC ENCRYPTION:" );
		LOGGER.info( theRuns + " / seconds = " + ((theEndTime - theStartTime) / 1000) );
		LOGGER.info( "1 / milliseconds = " + (((double) theEndTime - (double) theStartTime) / theRuns) );

		// DECRYPTION:
		theStartTime = System.currentTimeMillis();
		for ( int i = 0; i < theRuns; i++ ) {
			theDecryptedText = CryptographyUtility.toDecryptedText( theEncryptedText );
		}
		theEndTime = System.currentTimeMillis();
		LOGGER.info( "SYMETRIC DECRYPTION:" );
		LOGGER.info( theRuns + " / seconds = " + ((theEndTime - theStartTime) / 1000) );
		LOGGER.info( "1 / milliseconds = " + (((double) theEndTime - (double) theStartTime) / theRuns) );
	}
}
