// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.date.impls;

import java.text.AttributedCharacterIterator;
import java.text.DateFormatSymbols;
import java.text.FieldPosition;
import java.text.NumberFormat;
import java.text.ParseException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

import org.apache.commons.lang.time.DateUtils;
import org.apache.commons.lang.time.FastDateFormat;

/**
 * This concurrent {@link ConcurrentDateFormat} can be casted to
 * {@link SimpleDateFormat} with additional thread safety which
 * {@link SimpleDateFormat} cannot provide.
 */
public class ConcurrentDateFormat extends SimpleDateFormat {

	private static final long serialVersionUID = 1L;

	private FastDateFormat _fastDateFormat;
	private String[] _parsePatterns;

	/**
	 * {@inheritDoc}
	 */
	public ConcurrentDateFormat() {
		super();
		_fastDateFormat = FastDateFormat.getInstance();
	}

	/**
	 * {@inheritDoc}
	 */
	public ConcurrentDateFormat( String pattern, Locale locale ) {
		super( pattern, locale );
		_fastDateFormat = FastDateFormat.getInstance( pattern, locale );
		_parsePatterns = new String[] {
			pattern
		};
	}

	/**
	 * {@inheritDoc}
	 */
	public ConcurrentDateFormat( String pattern ) {
		super( pattern );
		_fastDateFormat = FastDateFormat.getInstance( pattern );
		_parsePatterns = new String[] {
			pattern
		};
	}

	// /////////////////////////////////////////////////////////////////////////
	// DELEGATES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 * 
	 * This method is overridden as the {@link #format(Date)} method is final
	 * and calls this method.
	 */
	@Override
	public StringBuffer format( Date date, StringBuffer toAppendTo, FieldPosition fieldPosition ) {
		return _fastDateFormat.format( date, toAppendTo, fieldPosition );
	}

	/**
	 * This methods is a substitute for the {@link #format(Date)} method which
	 * (in this case) unnecessarily calls the
	 * {@link #format(Date, StringBuffer, FieldPosition)} method with the result
	 * of creating unnecessary {@link StringBuffer} instances; please use this
	 * substitute of {@link #format(Date)}.
	 * 
	 * @param aDate The date to be formatted according to the given date.
	 * 
	 * @return The string as of the provided date.
	 * 
	 */
	public String toString( Date aDate ) {
		return _fastDateFormat.format( aDate );
	}

	/**
	 * {@inheritDoc}
	 * 
	 * Delegate the parse method to the thread safe {@link FastDateFormat}
	 * class.
	 */
	@Override
	public Date parse( String source ) throws ParseException {
		return DateUtils.parseDate( source, _parsePatterns );
	}

	// /////////////////////////////////////////////////////////////////////////
	// THREAD SAFETY:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 * 
	 * The {@link FastDateFormat} does not implement its
	 * {@link FastDateFormat#parseObject(String)} nor the
	 * {@link FastDateFormat#parseObject(String, ParsePosition)} methods :-(
	 * Therefore we only can make this method thread safe by synchronizing.
	 * 
	 * @deprecated as it is only thread safe due to synchronization.
	 */
	@Deprecated
	@Override
	public synchronized Date parse( String text, ParsePosition pos ) {
		return super.parse( text, pos );
	}

	// /////////////////////////////////////////////////////////////////////////
	// DEPRECATED:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public void set2DigitYearStart( Date startDate ) {
		super.set2DigitYearStart( startDate );
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public Date get2DigitYearStart() {
		return super.get2DigitYearStart();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public AttributedCharacterIterator formatToCharacterIterator( Object obj ) {
		return super.formatToCharacterIterator( obj );
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public void applyPattern( String pattern ) {
		super.applyPattern( pattern );
		_fastDateFormat = FastDateFormat.getInstance( pattern );
		_parsePatterns = new String[] {
			pattern
		};
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public void applyLocalizedPattern( String pattern ) {
		super.applyLocalizedPattern( pattern );
		_fastDateFormat = FastDateFormat.getInstance( pattern );
		_parsePatterns = new String[] {
			pattern
		};
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public DateFormatSymbols getDateFormatSymbols() {
		return super.getDateFormatSymbols();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public void setDateFormatSymbols( DateFormatSymbols newFormatSymbols ) {

		super.setDateFormatSymbols( newFormatSymbols );
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public Object parseObject( String source, ParsePosition pos ) {
		return super.parseObject( source, pos );
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public void setCalendar( Calendar newCalendar ) {
		super.setCalendar( newCalendar );
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public Calendar getCalendar() {
		return super.getCalendar();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public void setNumberFormat( NumberFormat newNumberFormat ) {
		super.setNumberFormat( newNumberFormat );
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public NumberFormat getNumberFormat() {
		return super.getNumberFormat();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public void setTimeZone( TimeZone zone ) {
		super.setTimeZone( zone );
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public TimeZone getTimeZone() {
		return super.getTimeZone();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public void setLenient( boolean lenient ) {
		super.setLenient( lenient );
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public boolean isLenient() {
		return super.isLenient();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated as it is not considered to be thread safe.
	 */
	@Deprecated
	@Override
	public Object parseObject( String source ) throws ParseException {
		return super.parseObject( source );
	}
}
