// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.graphical;

import java.io.InputStream;
import java.net.URL;

import javafx.scene.image.Image;
import javafx.scene.image.PixelReader;

/**
 * The Class RgbPixmapImageBuilderImpl.
 *
 * @author steiner
 */
public class RgbPixmapImageBuilderImpl implements RgbPixmapImageBuilder {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private InputStream _imageStream = null;

	private URL _imageUrl = null;

	private int _width = -1;

	private int _height = -1;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getWidth() {
		return _width;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getHeight() {
		return _height;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDimension( int aWidth, int aHeight ) {
		_width = aWidth;
		_height = aHeight;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDimension( Dimension aDimension ) {
		_width = aDimension.getWidth();
		_height = aDimension.getHeight();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setWidth( int aWidth ) {
		_width = aWidth;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setHeight( int aHeight ) {
		_height = aHeight;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public RgbPixmapImageBuilder withDimension( int aWidth, int aHeight ) {
		setDimension( aWidth, aHeight );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public RgbPixmapImageBuilder withDimension( Dimension aDimension ) {
		setDimension( aDimension );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public RgbPixmapImageBuilder withWidth( int aWidth ) {
		setWidth( aWidth );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public RgbPixmapImageBuilder withHeight( int aHeight ) {
		setHeight( aHeight );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setImageInputStream( InputStream aImageStream ) {
		_imageStream = aImageStream;
		_imageUrl = null;

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setImageURL( URL aImageUrl ) {
		_imageStream = null;
		_imageUrl = aImageUrl;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public RgbPixmap toPixmap() {
		if ( _imageUrl != null ) return toPixmap( new Image( _imageUrl.toString(), _width, _height, true, false ) );
		if ( _imageStream != null ) return toPixmap( new Image( _imageStream, _width, _height, true, false ) );
		throw new IllegalStateException( "Either an image URL or an image Input-Stream (File) must be set to produce a pixmap!" );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public RgbPixmap toPixmap( URL aImageUrl ) {
		return toPixmap( new Image( aImageUrl.toString(), _width, _height, true, false ) );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public RgbPixmap toPixmap( InputStream aImageStream ) {
		return toPixmap( new Image( aImageStream, _width, _height, true, false ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * To pixmap.
	 *
	 * @param aImage the image
	 * @return the rgb pixmap
	 */
	protected static RgbPixmap toPixmap( Image aImage ) {
		int theWidth = (int) aImage.getWidth();
		int theHeight = (int) aImage.getHeight();
		RgbPixel[][] thePixels = new RgbPixel[theWidth][theHeight];
		RgbPixmap thePixmap = new RgbPixmapImpl( thePixels );
		PixelReader theReader = aImage.getPixelReader();
		for ( int x = 0; x < theWidth; x++ ) {
			for ( int y = 0; y < theHeight; y++ ) {
				thePixels[x][y] = new RgbPixelImpl( theReader.getArgb( x, y ) );
			}
		}
		return thePixmap;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
