// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import org.refcodes.component.CloseException;
import org.refcodes.component.OpenException;

/**
 * Implementation of the {@link LoopbackShortSender} interface.
 */
public class LoopbackShortSenderImpl extends AbstractShortSender implements LoopbackShortSender {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private LoopbackShortReceiver _loopbackReceiver;

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isOpenable( LoopbackShortReceiver aLoopbackReceiver ) {
		return !isOpened() && aLoopbackReceiver != null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void open( LoopbackShortReceiver aLoopbackReceiver ) throws OpenException {
		if ( isOpened() ) {
			if ( _loopbackReceiver == aLoopbackReceiver && _loopbackReceiver.isOpened() ) {
				return;
			}
			throw new OpenException( "Unable to open the connection is is is ALREADY OPEN; connection status is " + getConnectionStatus() + "." );
		}
		super.open();
		_loopbackReceiver = aLoopbackReceiver;
		if ( _loopbackReceiver.isOpenable( this ) ) {
			_loopbackReceiver.open( this );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void writeDatagram( short aDatagram ) throws OpenException {
		if ( !isOpened() ) {
			throw new OpenException( "Unable to write datagram <" + aDatagram + "> as the connection is NOT OPEN; connection status is " + getConnectionStatus() + "." );
		}
		_loopbackReceiver.pushDatagram( aDatagram );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void writeDatagrams( short[] aDatagrams, int aOffset, int aLength ) throws OpenException {
		if ( !isOpened() ) {
			throw new OpenException( "Unable to write datagram <" + aDatagrams + "> as the connection is NOT OPEN; connection status is " + getConnectionStatus() + "." );
		}
		_loopbackReceiver.pushDatagrams( aDatagrams, aOffset, aLength );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void writeDatagrams( short[] aDatagrams ) throws OpenException {
		if ( !isOpened() ) {
			throw new OpenException( "Unable to write datagram <" + aDatagrams + "> as the connection is NOT OPEN; connection status is " + getConnectionStatus() + "." );
		}
		_loopbackReceiver.pushDatagrams( aDatagrams );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void flush() throws OpenException {
		// Nothing to do as datagrams are pushed directly to loopback receiver
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void close() throws CloseException {
		super.close();
		if ( _loopbackReceiver != null && !_loopbackReceiver.isClosed() ) {
			_loopbackReceiver.close();
			_loopbackReceiver = null;
		}
	}
}
