// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectStreamClass;
import java.io.StreamCorruptedException;

/**
 * The {@link SerializableObjectInputStreamImpl} extends an ordinary
 * {@link ObjectInputStream} to also resolve primitive types. In former JDKs
 * (not tested with current ones), primitive types caused problems when
 * deserializing. This class addresses and patches this issue.
 * <p>
 * Interesting enough that there were already virtual class definitions for the
 * primitive types ("boolean.class" - {@link Boolean#TYPE}, "byte.class" -
 * {@link Byte#TYPE}, "char.clas" - {@link Character#TYPE}, "int.class" -
 * {@link Integer#TYPE}, "double.class" - {@link Double#TYPE}, "long.class" -
 * {@link Long#TYPE}, float.class" - {@link Float#TYPE}).
 */
public class SerializableObjectInputStreamImpl extends ObjectInputStream {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new serializable object input stream impl.
	 *
	 * @param in the in
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws StreamCorruptedException the stream corrupted exception
	 */
	public SerializableObjectInputStreamImpl( InputStream in ) throws IOException, StreamCorruptedException {
		super( in );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Class<?> resolveClass( ObjectStreamClass aClass ) throws ClassNotFoundException, IOException {
		try {
			return super.resolveClass( aClass );
		}
		catch ( ClassNotFoundException aException ) {
			if ( aClass.getName().equals( "boolean" ) ) {
				return boolean.class;
			}
			else if ( aClass.getName().equals( "byte" ) ) {
				return byte.class;
			}
			else if ( aClass.getName().equals( "char" ) ) {
				return char.class;
			}
			else if ( aClass.getName().equals( "double" ) ) {
				return double.class;
			}
			else if ( aClass.getName().equals( "float" ) ) {
				return float.class;
			}
			else if ( aClass.getName().equals( "int" ) ) {
				return int.class;
			}
			else if ( aClass.getName().equals( "long" ) ) {
				return long.class;
			}
			else if ( aClass.getName().equals( "short" ) ) {
				return short.class;
			}
			else if ( aClass.getName().equals( "void" ) ) {
				return void.class;
			}
			throw aException;
		}
	}
}
