// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import java.io.Serializable;

import org.refcodes.component.OpenException;

/**
 * The {@link BlockProvider} is used to receive datagram blocks (arrays) in a
 * unified way. The {@link #readDatagrams()} method provides the next available
 * datagram block from the counterpart {@link BlockProvider} or
 * {@link DatagramProvider}; in case there is none available, then this method
 * halts until one is available.
 *
 * @param <DATA> The type of the datagram block (array) to be operated with. Do
 *        not provide an array type as the methods use to generic type for
 *        defining an array argument.
 */
@FunctionalInterface
public interface BlockProvider<DATA extends Serializable> {

	/**
	 * Reads (receives) the next datagram block passed from a
	 * {@link BlockSender} or {@link DatagramSender} counterpart. In case none
	 * datagram block is available, then this method blocks until one is
	 * available.
	 * 
	 * When a {@link Thread} is waiting for a datagram to be read and
	 * {@link Thread#interrupt()} is being called, then the operation is aborted
	 * and an {@link InterruptedException} is thrown.
	 * 
	 * @return The next datagram block sent from the {@link BlockSender} or
	 *         {@link DatagramReceiver} counterpart.
	 * 
	 * @throws OpenException Thrown in case opening or accessing an open line
	 *         (connection, junction, link) caused problems.
	 * 
	 * @throws InterruptedException Thrown when a {@link Thread} is waiting,
	 *         sleeping, or otherwise occupied, and the {@link Thread} is
	 *         interrupted, either before or during the activity.
	 */
	default DATA[] readDatagrams() throws OpenException, InterruptedException {
		return readDatagrams( -1 );
	}

	/**
	 * Similar to {@link #readDatagrams()} though at maximum the amount of data
	 * as provided by the block-size is returned.
	 * 
	 * @param aBlockSize The block-size which is not to exceeded by the returned
	 *        data. A value of -1 specifies to retrieve all available datagrams
	 *        (same behavior as method {@link #readDatagrams()}.
	 * 
	 * @return The next short block sent from the {@link BlockSender} or
	 *         {@link DatagramReceiver} counterpart.
	 * @throws OpenException Thrown in case opening or accessing an open line
	 *         (connection, junction, link) caused problems.
	 * 
	 * @throws InterruptedException Thrown when a {@link Thread} is waiting,
	 *         sleeping, or otherwise occupied, and the {@link Thread} is
	 *         interrupted, either before or during the activity.
	 */
	DATA[] readDatagrams( int aBlockSize ) throws OpenException, InterruptedException;

}
