// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import org.refcodes.component.OpenException;

/**
 * A {@link LoopbackShortReceiver} is a {@link DatagramReceiver} connected
 * directly within the same JVM with a {@link LoopbackShortSender}; a "loopback"
 * connection is used for establishing the connection which cannot be accessed
 * outside the JVM or the running machine.
 */
public interface LoopbackShortReceiver extends ConnectionShortReceiver<LoopbackShortSender> {

	/**
	 * Pushes a datagram into the {@link LoopbackShortReceiver}. Them datagrams
	 * can be retrieved via {@link #readDatagram()}: use {@link #hasDatagram()}
	 * to test beforehand whether there is a datagram available.
	 *
	 * @param aDatagram The datagram to be pushed into the
	 *        {@link LoopbackShortReceiver}; to be retrieved with the
	 *        {@link #readDatagram()} method.
	 * @throws OpenException the open exception
	 */
	void pushDatagram( short aDatagram ) throws OpenException;

	/**
	 * Pushes the datagrams into the {@link LoopbackByteReceiver}. Them
	 * datagrams can be retrieved via {@link #readDatagram()}: use
	 * {@link #hasDatagram()} to test beforehand whether there is a datagram
	 * available.
	 *
	 * @param aDatagrams The datagrams to be pushed into the
	 *        {@link LoopbackByteReceiver}; to be retrieved with the
	 *        {@link #readDatagram()} method.
	 * @throws OpenException the open exception
	 */
	default void pushDatagrams( short[] aDatagrams ) throws OpenException {
		pushDatagrams( aDatagrams, 0, aDatagrams.length );
	}

	/**
	 * Pushes the according number of datagrams beginning at the given offset
	 * into the {@link LoopbackByteReceiver}. Them datagrams can be retrieved
	 * via {@link #readDatagram()}: use {@link #hasDatagram()} to test
	 * beforehand whether there is a datagram available.
	 *
	 * @param aDatagrams The datagrams to be pushed into the
	 *        {@link LoopbackByteReceiver}; to be retrieved with the
	 *        {@link #readDatagram()} method.
	 * @param aOffset The offset from which to begin pushing.
	 * @param aLength The number of elements to be pushed.
	 * @throws OpenException the open exception
	 */
	default void pushDatagrams( short[] aDatagrams, int aOffset, int aLength ) throws OpenException {
		for ( int i = 0; i < aLength; i++ ) {
			pushDatagram( aDatagrams[i + aOffset] );
		}
	}

}
