// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.observer;

import org.refcodes.matcher.Matcher;
import org.refcodes.matcher.MatcherSugar;

/**
 * Declarative syntactic sugar which may be statically imported in order to
 * allow declarative definitions for the {@link EventMatcher} elements.
 */
public class EventMatcherSugar {

	/**
	 * Catches all events, no matching is done.
	 * 
	 * @return The "catch-all" {@link EventMatcher}.
	 * 
	 * @param <E> The type of the event to be matched
	 */
	public static <E extends GenericMetaDataEvent<?, ?>> EventMatcher<E> catchAll() {
		return new EventMatcher<E>() {

			@Override
			public boolean isMatching( E aEvent ) {
				return true;
			}
		};
	}

	/**
	 * Catches no event, no matching is done.
	 * 
	 * @return The "catch-none" {@link EventMatcher}.
	 * 
	 * @param <E> The type of the event to be matched
	 */
	public static <E extends GenericMetaDataEvent<?, ?>> EventMatcher<E> catchNone() {
		return new EventMatcher<E>() {

			@Override
			public boolean isMatching( E aEvent ) {
				return false;
			}
		};
	}

	/**
	 * Factory method to create an event matcher by event type.
	 * 
	 * @param <E> The type of the event to be matched
	 * 
	 * @param aEventType The event type to be matched by this matcher.
	 * 
	 * @return An event matcher by event type.
	 */
	public static <E extends GenericMetaDataEvent<?, ?>> EventMatcher<E> isAssignableFrom( Class<?> aEventType ) {
		Matcher<E> theAssignableFrom = MatcherSugar.isAssignableFrom( aEventType );
		return new EventMatcherWrapper<E>( theAssignableFrom );
	}

	/**
	 * Factory method to create an event matcher by event publisher type.
	 * 
	 * @param <E> The type of the event to be matched
	 * @param <PT> The publisher descriptor type
	 * 
	 * @param aPublisherType The event publisher type to be matched by this
	 *        matcher.
	 * 
	 * @return An event matcher by event type.
	 */
	public static <E extends GenericMetaDataEvent<?, ?>, PT extends Object> EventMatcher<E> publisherIsAssignableFrom( Class<? extends PT> aPublisherType ) {
		return new PublisherIsAssignableFromMatcherImpl<E, PT>( aPublisherType );
	}

	/**
	 * Factory method to create an "OR" matcher for the given matchers.
	 * 
	 * @param <E> The event type applied to the matcher
	 * 
	 * @param aEventMatchers The matchers to be combined by an "OR".
	 * 
	 * @return An "OR" matcher.
	 */
	@SafeVarargs
	public static <E extends GenericMetaDataEvent<?, ?>> EventMatcher<E> or( EventMatcher<E>... aEventMatchers ) {
		return new EventMatcherWrapper<E>( MatcherSugar.or( aEventMatchers ) );
	}

	/**
	 * Factory method to create an "AND" matcher for the given matchers.
	 * 
	 * @param <E> The event type applied to the matcher
	 * 
	 * @param aEventMatchers The matchers to be combined by an "AND".
	 * 
	 * @return An "AND" matcher.
	 */
	@SafeVarargs
	public static <E extends GenericMetaDataEvent<?, ?>> EventMatcher<E> and( EventMatcher<E>... aEventMatchers ) {
		return new EventMatcherWrapper<E>( MatcherSugar.and( aEventMatchers ) );
	}

	/**
	 * Factory method to create an "EQUAL WITH" matcher for the given name
	 * compared with the name stored in the {@link EventMetaData}.
	 * 
	 * @param <E> The event type applied to the matcher
	 * 
	 * @param aAlias The alias to be compared with a
	 *        {@link GenericMetaDataEvent}'s {@link EventMetaData}'s name
	 *        property.
	 * 
	 * @return An "EQUAL WITH" matcher regarding the
	 *         {@link GenericMetaDataEvent}'s name property.
	 */
	public static <E extends GenericMetaDataEvent<?, ?>> EventMatcher<E> nameEqualWith( String aAlias ) {
		return new AliasEqualWithMatcherImpl<E>( aAlias );
	}

	/**
	 * Factory method to create an "EQUAL WITH" matcher for the given group
	 * compared with the group stored in the {@link EventMetaData}.
	 * 
	 * @param <E> The event type applied to the matcher
	 * 
	 * @param aGroup The group to be compared with a
	 *        {@link GenericMetaDataEvent}'s {@link EventMetaData}'s group
	 *        property.
	 * 
	 * @return An "EQUAL WITH" matcher regarding the
	 *         {@link GenericMetaDataEvent}'s group property.
	 */
	public static <E extends GenericMetaDataEvent<?, ?>> EventMatcher<E> groupEqualWith( String aGroup ) {
		return new GroupEqualWithMatcherImpl<E>( aGroup );
	}

	/**
	 * Factory method to create an "EQUAL WITH" matcher for the given UID
	 * compared with the UID stored in the {@link EventMetaData}.
	 * 
	 * @param <E> The event type applied to the matcher
	 * 
	 * @param aUid The UID to be compared with a {@link GenericMetaDataEvent}'s
	 *        {@link EventMetaData}'s UID property.
	 * 
	 * @return An "EQUAL WITH" matcher regarding the
	 *         {@link GenericMetaDataEvent}'s UID property.
	 */
	public static <E extends GenericMetaDataEvent<?, ?>> EventMatcher<E> universalIdEqualWith( String aUid ) {
		return new UniversalIdEqualWithMatcherImpl<E>( aUid );
	}

	/**
	 * Factory method to create an "EQUAL WITH" matcher for the given action
	 * compared with the action stored in the {@link EventMetaData}.
	 * 
	 * @param <E> The event type applied to the matcher.
	 * 
	 * @param <A> The type of the action stored in the event. CAUTION: The
	 *        drawback of not using generic generic type declaration on a class
	 *        level is no granted type safety, the advantage is the ease of use:
	 *        Sub-classes can be used out of the box.
	 * 
	 * @param aAction The action to be compared with a
	 *        {@link GenericMetaDataEvent}'s {@link EventMetaData}'s action
	 *        property.
	 * 
	 * @return An "EQUAL WITH" matcher regarding the
	 *         {@link GenericActionEvent}'s action property.
	 */
	public static <E extends GenericActionMetaDataEvent<A, ?, ?>, A> EventMatcher<E> actionEqualWith( A aAction ) {
		return new ActionEqualWithMatcherImpl<E>( aAction );
	}

	// -------------------------------------------------------------------------
	// INNER CLASSES:
	// -------------------------------------------------------------------------

	/**
	 * Wrapps an event matcher to by of the correct type.
	 *
	 * @param <E> the element type
	 */
	private static class EventMatcherWrapper<E extends GenericMetaDataEvent<?, ?>> implements EventMatcher<E> {

		private Matcher<E> _eventMatcher;

		/**
		 * Wrapps the given event matcher to be of the correct type.
		 * 
		 * @param aMatcher the event matcher in question.
		 */
		public EventMatcherWrapper( Matcher<E> aMatcher ) {
			assert (aMatcher != null);
			_eventMatcher = aMatcher;
		}

		/**
		 * Checks if is matching.
		 *
		 * @param aEvent the event
		 * @return true, if is matching
		 */
		@Override
		public boolean isMatching( E aEvent ) {
			return _eventMatcher.isMatching( aEvent );
		}
	}

	/**
	 * A basic implementation of an event matcher.
	 *
	 * @param <E> The event type
	 * @param <PT> the generic type
	 */
	private static class PublisherIsAssignableFromMatcherImpl<E extends GenericMetaDataEvent<?, ?>, PT extends Object> implements EventMatcher<E> {

		private Class<? extends PT> _eventPublisherType;

		/**
		 * Instantiates a new publisher is assignable from matcher impl.
		 *
		 * @param aEventPublisherType the event publisher type
		 */
		public PublisherIsAssignableFromMatcherImpl( Class<? extends PT> aEventPublisherType ) {
			_eventPublisherType = aEventPublisherType;
		}

		/**
		 * Checks if is matching.
		 *
		 * @param aEvent the event
		 * @return true, if is matching
		 */
		@Override
		public boolean isMatching( E aEvent ) {
			assert (aEvent != null);
			if ( _eventPublisherType != null ) {
				if ( !aEvent.getMetaData().getPublisherType().isAssignableFrom( _eventPublisherType ) ) {
					return false;
				}
			}
			return true;
		}
	}

	/**
	 * A basic implementation of an event matcher.
	 * 
	 * @param <E> The event type
	 */
	private static class AliasEqualWithMatcherImpl<E extends GenericMetaDataEvent<?, ?>> implements EventMatcher<E> {

		private String _alias;

		/**
		 * Instantiates a new alias equal with matcher impl.
		 *
		 * @param aAlias the alias
		 */
		public AliasEqualWithMatcherImpl( String aAlias ) {
			_alias = aAlias;
		}

		/**
		 * Checks if is matching.
		 *
		 * @param aEvent the event
		 * @return true, if is matching
		 */
		@Override
		public boolean isMatching( E aEvent ) {
			assert (aEvent != null);

			if ( _alias != null ) {
				return (_alias.equals( aEvent.getMetaData().getAlias() ));
			}
			else if ( aEvent.getMetaData().getAlias() == null ) {
				return true;
			}
			return false;
		}
	}

	/**
	 * A basic implementation of an event matcher.
	 * 
	 * @param <E> The event type
	 */
	private static class GroupEqualWithMatcherImpl<E extends GenericMetaDataEvent<?, ?>> implements EventMatcher<E> {

		private String _group;

		/**
		 * Instantiates a new group equal with matcher impl.
		 *
		 * @param aGroup the group
		 */
		public GroupEqualWithMatcherImpl( String aGroup ) {
			_group = aGroup;
		}

		/**
		 * Checks if is matching.
		 *
		 * @param aEvent the event
		 * @return true, if is matching
		 */
		@Override
		public boolean isMatching( E aEvent ) {
			assert (aEvent != null);

			if ( _group != null ) {
				return (_group.equals( aEvent.getMetaData().getGroup() ));
			}
			else if ( aEvent.getMetaData().getGroup() == null ) {
				return true;
			}
			return false;
		}
	}

	/**
	 * A basic implementation of an event matcher.
	 * 
	 * @param <E> The event type
	 */
	private static class UniversalIdEqualWithMatcherImpl<E extends GenericMetaDataEvent<?, ?>> implements EventMatcher<E> {

		private String _universalId;

		/**
		 * Instantiates a new Universal-ID equal with matcher impl.
		 *
		 * @param aUid the Universal-ID
		 */
		public UniversalIdEqualWithMatcherImpl( String aUid ) {
			_universalId = aUid;
		}

		/**
		 * Checks if is matching.
		 *
		 * @param aEvent the event
		 * @return true, if is matching
		 */
		@Override
		public boolean isMatching( E aEvent ) {
			assert (aEvent != null);

			if ( _universalId != null ) {
				return (_universalId.equals( aEvent.getMetaData().getUniversalId() ));
			}
			else if ( aEvent.getMetaData().getUniversalId() == null ) {
				return true;
			}
			return false;
		}
	}

	/**
	 * A basic implementation of an event matcher.
	 * 
	 * @param <E> The event type
	 */
	private static class ActionEqualWithMatcherImpl<E extends GenericActionMetaDataEvent<?, ?, ?>> implements EventMatcher<E> {

		private Object _action;

		/**
		 * Instantiates a new action equal with matcher impl.
		 *
		 * @param <A> the generic type
		 * @param aAction the action
		 */
		public <A> ActionEqualWithMatcherImpl( A aAction ) {
			_action = aAction;
		}

		/**
		 * Checks if is matching.
		 *
		 * @param aEvent the event
		 * @return true, if is matching
		 */
		@Override
		public boolean isMatching( E aEvent ) {
			assert (aEvent != null);

			if ( _action != null ) {
				return (_action.equals( aEvent.getAction() ));
			}
			else if ( aEvent.getAction() == null ) {
				return true;
			}
			return false;
		}
	}
}
