// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import org.refcodes.data.Delimiter;
import org.refcodes.data.Protocol;
import org.refcodes.net.FormFields;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.MediaType;
import org.refcodes.net.MediaTypeFactory;
import org.refcodes.net.MediaTypeFactoryLookup.MutableMediaTypeFactoryLookup;
import org.refcodes.net.Port;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.UserAgentAccessor.UserAgentBuilder;
import org.refcodes.net.UserAgentAccessor.UserAgentProperty;
import org.refcodes.net.impls.ApplicationFormFactory;
import org.refcodes.net.impls.ApplicationJsonFactory;
import org.refcodes.net.impls.ApplicationXmlFactory;
import org.refcodes.net.impls.TextPlainFactory;
import org.refcodes.rest.impls.HttpRestClientImpl;
import org.refcodes.runtime.RequestCorrelation;
import org.refcodes.runtime.SessionCorrelation;

/**
 * A client to send requests for communicating with a RESTful server such as the
 * {@link HttpRestServer}.
 * 
 * For marshaling and unmarshaling HTTP bodies you have to register the
 * according {@link MediaTypeFactory} instances being capable of handling the
 * according {@link MediaType} definitions. Examples for ready to use
 * {@link MediaTypeFactory} implementations are
 * 
 * <ul>
 * <li>{@link ApplicationJsonFactory},
 * <li>{@link ApplicationXmlFactory}
 * <li>{@link ApplicationFormFactory}
 * <li>{@link TextPlainFactory}
 * </ul>
 * 
 * Implementations of the {@link RestClient} interface, such as the
 * {@link HttpRestClientImpl}, are initialized with some common
 * {@link MediaTypeFactory} instances.
 * 
 * You may set an individual User-Agent via {@link #setUserAgent(String)}, not
 * setting it will result in the default User-Agent to be set as coded into the
 * {@link RestClient} implementations.
 */
public interface RestClient extends MutableMediaTypeFactoryLookup, UserAgentProperty, UserAgentBuilder<RestClient>, RequestCorrelation<RestClient>, SessionCorrelation<RestClient> {

	/**
	 * With user agent.
	 *
	 * @param aUserAgent the a user agent
	 * @return the rest client
	 */
	@Override
	default RestClient withUserAgent( String aUserAgent ) {
		setUserAgent( aUserAgent );
		return this;
	}

	/**
	 * With session correlation.
	 *
	 * @param hasSessionCorrelation the has session correlation
	 * @return the rest client
	 */
	@Override
	default RestClient withSessionCorrelation( boolean hasSessionCorrelation ) {
		setSessionCorrelation( hasSessionCorrelation );
		return this;
	}

	/**
	 * With enable session correlation.
	 *
	 * @return the rest client
	 */
	@Override
	default RestClient withEnableSessionCorrelation() {
		enableSessionCorrelation();
		return this;
	}

	/**
	 * With disable session correlation.
	 *
	 * @return the rest client
	 */
	@Override
	default RestClient withDisableSessionCorrelation() {
		disableSessionCorrelation();
		return this;
	}

	/**
	 * With request correlation.
	 *
	 * @param hasRequestCorrelation the has request correlation
	 * @return the rest client
	 */
	@Override
	default RestClient withRequestCorrelation( boolean hasRequestCorrelation ) {
		setRequestCorrelation( hasRequestCorrelation );
		return this;
	}

	/**
	 * With enable request correlation.
	 *
	 * @return the rest client
	 */
	@Override
	default RestClient withEnableRequestCorrelation() {
		enableRequestCorrelation();
		return this;
	}

	/**
	 * With disable request correlation.
	 *
	 * @return the rest client
	 */
	@Override
	default RestClient withDisableRequestCorrelation() {
		disableRequestCorrelation();
		return this;
	}

	// /////////////////////////////////////////////////////////////////////////
	// REQUEST BUILDER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestRequestBuilder} at this request.
	 */
	RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest );

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aLocator, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( aHttpMethod, aLocator, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( aHttpMethod, aLocator, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aLocator, Object aRequest ) {
		return buildRequest( aHttpMethod, aLocator, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields ) {
		return buildRequest( aHttpMethod, aLocator, aQueryFields, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aLocator ) {
		return buildRequest( aHttpMethod, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( String aLocator, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.GET, aLocator, aQueryFields, null, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.GET, aLocator, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( String aLocator, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.GET, aLocator, null, aHeaderFields, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( String aLocator ) {
		return buildRequest( HttpMethod.GET, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aLocator, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aLocator, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aLocator, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aLocator, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aLocator, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.POST, aLocator, aQueryFields, null, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aLocator, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aLocator, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aLocator, null, aHeaderFields, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aLocator, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aLocator, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aLocator ) {
		return buildRequest( HttpMethod.POST, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aLocator, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aLocator, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aLocator, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aLocator, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aLocator, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.DELETE, aLocator, aQueryFields, null, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aLocator, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.DELETE, aLocator, null, aHeaderFields, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.DELETE, aLocator, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aLocator, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aLocator, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aLocator ) {
		return buildRequest( HttpMethod.DELETE, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aLocator, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aLocator, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aLocator, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aLocator, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aLocator, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.PUT, aLocator, aQueryFields, null, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aLocator, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.PUT, aLocator, null, aHeaderFields, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.PUT, aLocator, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aLocator, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aLocator, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aLocator ) {
		return buildRequest( HttpMethod.PUT, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	// /////////////////////////////////////////////////////////////////////////
	// SYNCHRONOUS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aRequest The {@link RestRequest} encapsulating all necessary
	 *        attributes to issue the request.
	 * 
	 * @return The {@link RestResponse} at this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( RestRequest aRequest ) throws HttpResponseException {
		return doRequest( aRequest.getHttpMethod(), aRequest.getLocator(), aRequest.getQueryFields(), aRequest.getHeaderFields(), (Object) aRequest.getRequest() );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestResponse} at this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException;

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aLocator, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aLocator, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aLocator, null, aHeaderFields, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aLocator, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aLocator, (FormFields) null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aLocator, aQueryFields, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aLocator ) throws HttpResponseException {
		return doRequest( aHttpMethod, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( String aLocator, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aLocator, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aLocator, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( String aLocator, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aLocator, null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( String aLocator ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aLocator, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aLocator, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aLocator, null, aHeaderFields, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aLocator, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aLocator, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aLocator, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aLocator, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aLocator, (FormFields) null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aLocator ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aLocator, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aLocator, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aLocator, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aLocator, null, aHeaderFields, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aLocator, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aLocator, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aLocator, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aLocator, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aLocator, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aLocator, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aLocator, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aLocator ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aLocator, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aLocator, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aLocator, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aLocator, aHeaderFields, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aLocator, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aLocator, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aLocator, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aLocator, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aLocator, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aLocator, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aLocator, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aLocator ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	// /////////////////////////////////////////////////////////////////////////
	// ASYNCHRONOUS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 * @return The {@link RestCaller} which is used by the request.
	 */
	RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver );

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aLocator, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aLocator, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aLocator, (FormFields) null, (RequestHeaderFields) null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aLocator, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aLocator, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aLocator, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( String aLocator, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aLocator, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( String aLocator, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aLocator, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aLocator, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( String aLocator, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aLocator, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aLocator, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aLocator, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aLocator, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aLocator, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aLocator, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aLocator, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aLocator, (FormFields) null, (RequestHeaderFields) null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aLocator, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aLocator, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aLocator, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aLocator, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aLocator, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aLocator, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aLocator, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aLocator, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aLocator, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aLocator, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aLocator, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aLocator, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aLocator, (FormFields) null, (RequestHeaderFields) null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aLocator, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aLocator, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aLocator, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aLocator, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aLocator, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aLocator, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aLocator, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aLocator, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 *
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aLocator, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aLocator, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aLocator, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aLocator, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aLocator, (FormFields) null, (RequestHeaderFields) null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The local locator to which this {@link RestCaller} is
	 *        bound.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aLocator, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aLocator, (FormFields) null, (RequestHeaderFields) null, (Object) null, aResponseObserver );
	}

	// /////////////////////////////////////////////////////////////////////////
	// PROTOCOL + HOST + PORT + PATH:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestRequestBuilder} at this request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		Protocol theProtocol = Protocol.fromName( aProtocol );
		if ( theProtocol == null ) throw new IllegalArgumentException( "The provided protocol <" + aProtocol + "> is not supported. Provide either \"HTTP\" or \"HTTPS\"." );
		int thePort = -1;
		if ( Protocol.HTTP == theProtocol )
			thePort = Port.HTTP.getValue();
		else if ( Protocol.HTTPS == theProtocol ) thePort = Port.HTTPS.getValue();
		if ( thePort == -1 ) throw new IllegalArgumentException( "You must provide either \"HTTP\" or \"HTTPS\" protocol, though you provided <" + aProtocol + ">." );
		if ( aPath != null ) {
			while ( aPath.startsWith( "" + Delimiter.PATH.getChar() ) ) {
				aPath = aPath.substring( 1 );
			}
		}
		String theLocator = theProtocol.getResourceLocatorPrefix() + aHost + Delimiter.PORT + thePort + Delimiter.PATH.getChar() + aPath;
		return buildRequest( aHttpMethod, theLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, Object aRequest ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPath, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPath, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( String aProtocol, String aHost, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( String aProtocol, String aHost, String aPath ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPath, null, null, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, null, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, null, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, null, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, String aPath ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, null, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, null, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, String aPath ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, null, null );
	}

	// /////////////////////////////////////////////////////////////////////////
	// SYNCHRONOUS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestResponse} at this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		Protocol theProtocol = Protocol.fromName( aProtocol );
		if ( theProtocol == null ) throw new IllegalArgumentException( "The provided protocol <" + aProtocol + "> is not supported. Provide either \"HTTP\" or \"HTTPS\"." );
		int thePort = -1;
		if ( Protocol.HTTP == theProtocol )
			thePort = Port.HTTP.getValue();
		else if ( Protocol.HTTPS == theProtocol ) thePort = Port.HTTPS.getValue();
		if ( thePort == -1 ) throw new IllegalArgumentException( "You must provide either \"HTTP\" or \"HTTPS\" protocol, though you provided <" + aProtocol + ">." );
		if ( aPath != null ) {
			while ( aPath.startsWith( "" + Delimiter.PATH.getChar() ) ) {
				aPath = aPath.substring( 1 );
			}
		}
		String theLocator = theProtocol.getResourceLocatorPrefix() + aHost + Delimiter.PORT + thePort + Delimiter.PATH.getChar() + aPath;
		return doRequest( aHttpMethod, theLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, (FormFields) null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( String aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( String aProtocol, String aHost, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, null, null, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, null, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, null, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, null, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, null, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, null, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, null, (Object) null );
	}

	// /////////////////////////////////////////////////////////////////////////
	// ASYNCHRONOUS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		Protocol theProtocol = Protocol.fromName( aProtocol );
		if ( theProtocol == null ) throw new IllegalArgumentException( "The provided protocol <" + aProtocol + "> is not supported. Provide either \"HTTP\" or \"HTTPS\"." );
		int thePort = -1;
		if ( Protocol.HTTP == theProtocol )
			thePort = Port.HTTP.getValue();
		else if ( Protocol.HTTPS == theProtocol ) thePort = Port.HTTPS.getValue();
		if ( thePort == -1 ) throw new IllegalArgumentException( "You must provide either \"HTTP\" or \"HTTPS\" protocol, though you provided <" + aProtocol + ">." );
		if ( aPath != null ) {
			while ( aPath.startsWith( "" + Delimiter.PATH.getChar() ) ) {
				aPath = aPath.substring( 1 );
			}
		}
		String theLocator = theProtocol.getResourceLocatorPrefix() + aHost + Delimiter.PORT + thePort + Delimiter.PATH.getChar() + aPath;
		return doRequest( aHttpMethod, theLocator, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( String aProtocol, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, null, null, aResponseObserver );
	}

	// /////////////////////////////////////////////////////////////////////////
	// PROTOCOL + HOST + PORT + PATH:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestRequestBuilder} at this request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		Protocol theProtocol = Protocol.fromName( aProtocol );
		if ( theProtocol == null ) throw new IllegalArgumentException( "The provided protocol <" + aProtocol + "> is not supported. Provide either \"HTTP\" or \"HTTPS\"." );
		if ( aPath != null ) {
			while ( aPath.startsWith( "" + Delimiter.PATH.getChar() ) ) {
				aPath = aPath.substring( 1 );
			}
		}
		String theLocator = theProtocol.getResourceLocatorPrefix() + aHost + Delimiter.PORT.getChar() + aPort + Delimiter.PATH.getChar() + aPath;
		return buildRequest( aHttpMethod, theLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( String aProtocol, String aHost, int aPort, String aPath ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, null, null, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, null, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, null, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, null, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( String aProtocol, String aHost, int aPort, String aPath ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, null, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, null, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( String aProtocol, String aHost, int aPort, String aPath ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, null, null );
	}

	// /////////////////////////////////////////////////////////////////////////
	// SYNCHRONOUS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestResponse} at this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		Protocol theProtocol = Protocol.fromName( aProtocol );
		if ( theProtocol == null ) throw new IllegalArgumentException( "The provided protocol <" + aProtocol + "> is not supported. Provide either \"HTTP\" or \"HTTPS\"." );
		if ( aPath != null ) {
			while ( aPath.startsWith( "" + Delimiter.PATH.getChar() ) ) {
				aPath = aPath.substring( 1 );
			}
		}
		String theLocator = theProtocol.getResourceLocatorPrefix() + aHost + Delimiter.PORT.getChar() + aPort + Delimiter.PATH.getChar() + aPath;
		return doRequest( aHttpMethod, theLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, (FormFields) null, (RequestHeaderFields) aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( String aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, null, null, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, null, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, null, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, null, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( String aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, null, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, null, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( String aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, null, (Object) null );
	}

	// /////////////////////////////////////////////////////////////////////////
	// ASYNCHRONOUS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		Protocol theProtocol = Protocol.fromName( aProtocol );
		if ( theProtocol == null ) throw new IllegalArgumentException( "The provided protocol <" + aProtocol + "> is not supported. Provide either \"HTTP\" or \"HTTPS\"." );
		if ( aPath != null ) {
			while ( aPath.startsWith( "" + Delimiter.PATH.getChar() ) ) {
				aPath = aPath.substring( 1 );
			}
		}
		String theLocator = theProtocol.getResourceLocatorPrefix() + aHost + Delimiter.PORT.getChar() + aPort + Delimiter.PATH.getChar() + aPath;
		return doRequest( aHttpMethod, theLocator, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( String aProtocol, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( String aProtocol, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( String aProtocol, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, null, null, aResponseObserver );
	}

	// /////////////////////////////////////////////////////////////////////////
	// PROTOCOL (ENUM) + HOST + PORT + PATH:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestRequestBuilder} at this request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		if ( aProtocol == null ) {
			if ( aPort == Port.HTTPS.getValue() )
				aProtocol = Protocol.HTTPS;
			else aProtocol = Protocol.HTTP;
		}
		if ( aPath != null ) {
			while ( aPath.startsWith( "" + Delimiter.PATH.getChar() ) ) {
				aPath = aPath.substring( 1 );
			}
		}
		String theLocator = aProtocol.getResourceLocatorPrefix() + aHost + Delimiter.PORT.getChar() + aPort + Delimiter.PATH.getChar() + aPath;
		return buildRequest( aHttpMethod, theLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, Object aRequest ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( Protocol aProtocol, String aHost, int aPort, String aPath ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, null, null, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, int aPort, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, null, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, int aPort, String aPath ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, null, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, int aPort, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, null, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, int aPort, String aPath ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, null, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, int aPort, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, null, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, int aPort, String aPath ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, null, null );
	}

	// /////////////////////////////////////////////////////////////////////////
	// SYNCHRONOUS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestResponse} at this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		if ( aProtocol == null ) {
			if ( aPort == Port.HTTPS.getValue() )
				aProtocol = Protocol.HTTPS;
			else aProtocol = Protocol.HTTP;
		}
		if ( aPath != null ) {
			while ( aPath.startsWith( "" + Delimiter.PATH.getChar() ) ) {
				aPath = aPath.substring( 1 );
			}
		}
		String theLocator = aProtocol.getResourceLocatorPrefix() + aHost + Delimiter.PORT.getChar() + aPort + Delimiter.PATH.getChar() + aPath;
		return doRequest( aHttpMethod, theLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, (FormFields) null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( Protocol aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, null, null, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, null, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, null, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, null, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, null, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, null, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, null, (Object) null );
	}

	// /////////////////////////////////////////////////////////////////////////
	// ASYNCHRONOUS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		if ( aProtocol == null ) {
			if ( aPort == Port.HTTPS.getValue() )
				aProtocol = Protocol.HTTPS;
			else aProtocol = Protocol.HTTP;
		}
		if ( aPath != null ) {
			while ( aPath.startsWith( "" + Delimiter.PATH.getChar() ) ) {
				aPath = aPath.substring( 1 );
			}
		}
		String theLocator = aProtocol.getResourceLocatorPrefix() + aHost + Delimiter.PORT.getChar() + aPort + Delimiter.PATH.getChar() + aPath;
		return doRequest( aHttpMethod, theLocator, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( Protocol aProtocol, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPort, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPort, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 *
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPort The port to be used when connecting to the host.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPort, aPath, null, null, null, aResponseObserver );
	}

	// /////////////////////////////////////////////////////////////////////////
	// PROTOCOL (ENUM) + HOST + PORT + PATH:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestRequestBuilder} at this request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		if ( aProtocol == null ) throw new IllegalArgumentException( "You must provide either \"HTTP\" or \"HTTPS\" protocol, though you provided <" + aProtocol + ">." );
		int thePort = -1;
		if ( Protocol.HTTP == aProtocol )
			thePort = Port.HTTP.getValue();
		else if ( Protocol.HTTPS == aProtocol ) thePort = Port.HTTPS.getValue();
		if ( thePort == -1 ) throw new IllegalArgumentException( "You must provide either \"HTTP\" or \"HTTPS\" protocol, though you provided <" + aProtocol + ">." );
		if ( aPath != null ) {
			while ( aPath.startsWith( "" + Delimiter.PATH.getChar() ) ) {
				aPath = aPath.substring( 1 );
			}
		}
		String theLocator = aProtocol.getResourceLocatorPrefix() + aHost + Delimiter.PORT + thePort + Delimiter.PATH.getChar() + aPath;
		return buildRequest( aHttpMethod, theLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, Object aRequest ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPath, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath ) {
		return buildRequest( aHttpMethod, aProtocol, aHost, aPath, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a GET request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildGet( Protocol aProtocol, String aHost, String aPath ) {
		return buildRequest( HttpMethod.GET, aProtocol, aHost, aPath, null, null, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, null, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Protocol aProtocol, String aHost, String aPath ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, null, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, null, aRequest );
	}

	/**
	 * Prepares a DELETE request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildDelete( Protocol aProtocol, String aHost, String aPath ) {
		return buildRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, null, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, null, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, aHeaderFields, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, null, aRequest );
	}

	/**
	 * Prepares a PUT request builder with the common attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPut( Protocol aProtocol, String aHost, String aPath ) {
		return buildRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, null, null );
	}

	// /////////////////////////////////////////////////////////////////////////
	// SYNCHRONOUS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestResponse} at this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		if ( aProtocol == null ) throw new IllegalArgumentException( "You must provide either \"HTTP\" or \"HTTPS\" protocol, though you provided <" + aProtocol + ">." );
		int thePort = -1;
		if ( Protocol.HTTP == aProtocol )
			thePort = Port.HTTP.getValue();
		else if ( Protocol.HTTPS == aProtocol ) thePort = Port.HTTPS.getValue();
		if ( thePort == -1 ) throw new IllegalArgumentException( "You must provide either \"HTTP\" or \"HTTPS\" protocol, though you provided <" + aProtocol + ">." );
		if ( aPath != null ) {
			while ( aPath.startsWith( "" + Delimiter.PATH.getChar() ) ) {
				aPath = aPath.substring( 1 );
			}
		}
		String theLocator = aProtocol.getResourceLocatorPrefix() + aHost + Delimiter.PORT + thePort + Delimiter.PATH.getChar() + aPath;
		return doRequest( aHttpMethod, theLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, (FormFields) null, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, (FormFields) null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException The HTTP response exception
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath ) throws HttpResponseException {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, (FormFields) null, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a GET request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doGet( Protocol aProtocol, String aHost, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, null, null, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, null, aRequest );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Protocol aProtocol, String aHost, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, null, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, aHeaderFields, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, null, aRequest );
	}

	/**
	 * Sends a DELETE request with the common attributes and returns the
	 * according request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doDelete( Protocol aProtocol, String aHost, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, null, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, null, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, null, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, null, aRequest );
	}

	/**
	 * Sends a PUT request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPut( Protocol aProtocol, String aHost, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, null, (Object) null );
	}

	// /////////////////////////////////////////////////////////////////////////
	// ASYNCHRONOUS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		if ( aProtocol == null ) throw new IllegalArgumentException( "You must provide either \"HTTP\" or \"HTTPS\" protocol, though you provided <" + aProtocol + ">." );
		int thePort = -1;
		if ( Protocol.HTTP == aProtocol )
			thePort = Port.HTTP.getValue();
		else if ( Protocol.HTTPS == aProtocol ) thePort = Port.HTTPS.getValue();
		if ( thePort == -1 ) throw new IllegalArgumentException( "You must provide either \"HTTP\" or \"HTTPS\" protocol, though you provided <" + aProtocol + ">." );
		if ( aPath != null ) {
			while ( aPath.startsWith( "" + Delimiter.PATH.getChar() ) ) {
				aPath = aPath.substring( 1 );
			}
		}
		String theLocator = aProtocol.getResourceLocatorPrefix() + aHost + Delimiter.PORT + thePort + Delimiter.PATH.getChar() + aPath;
		return doRequest( aHttpMethod, theLocator, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestCaller} is
	 *        bound.
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Protocol aProtocol, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aProtocol, aHost, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a GET request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doGet( Protocol aProtocol, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.GET, aProtocol, aHost, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPost( Protocol aProtocol, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (a server may ignore
	 *        it when doing a {@link HttpMethod#DELETE} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a DELETE request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doDelete( Protocol aProtocol, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.DELETE, aProtocol, aHost, aPath, null, null, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, null, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 *
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#PUT} request.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, null, aRequest, aResponseObserver );
	}

	/**
	 * Creates a PUT request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The {@link Protocol} (e.g. HTTP or HTTPS) to be used for
	 *        the destination URL.
	 * @param aHost The host to which the destination URL is to point to.
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCaller} which is used by the request.
	 */
	default RestCallerBuilder doPut( Protocol aProtocol, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.PUT, aProtocol, aHost, aPath, null, null, null, aResponseObserver );
	}
}
