// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import org.refcodes.mixin.LocatorAccessor.LocatorBuilder;
import org.refcodes.mixin.LocatorAccessor.LocatorProperty;
import org.refcodes.net.FormFields;
import org.refcodes.net.HeaderFieldsAccessor.HeaderFieldsBuilder;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpMethodAccessor.HttpMethodBuilder;
import org.refcodes.net.HttpMethodAccessor.HttpMethodProperty;
import org.refcodes.net.HttpRequestBuilder;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.QueryFieldsAccessor.QueryFieldsBuilder;
import org.refcodes.net.RequestHeaderFields;

/**
 * An {@link RestRequestBuilder} extends an {@link RestCaller} with builder
 * functionality and adds <code>lambda</code> support for handling the responses
 * addressed to this {@link RestCaller}. The <code>lambda</code> defined as
 * {@link RestResponseObserver} acts as the single listener to this
 * {@link RestCaller} responsible for handling the responses for which this
 * {@link RestCaller} is responsible.
 * 
 * The locator to which a {@link RestRequestBuilder} targets for is defined by
 * the {@link #getLocator()} property.
 */
public interface RestRequestBuilder extends RestRequest, HttpRequestBuilder<RestRequestBuilder>, LocatorProperty<String>, LocatorBuilder<String, RestRequestBuilder>, HttpMethodProperty, HttpMethodBuilder<RestRequestBuilder>, QueryFieldsBuilder<RestRequestBuilder>, HeaderFieldsBuilder<RequestHeaderFields, RestRequestBuilder> {

	/**
	 * Sends the request and returns synchronously the according
	 * {@link RestResponse}.
	 *
	 * @return the rest response
	 * @throws HttpResponseException the http response exception
	 */
	RestResponse toRestResponse() throws HttpResponseException;

	/**
	 * With query fields.
	 *
	 * @param aQueryFields the a query fields
	 * @return the rest request builder
	 */
	@Override
	default RestRequestBuilder withQueryFields( FormFields aQueryFields ) {
		setQueryFields( aQueryFields );
		return this;
	}

	/**
	 * With locator.
	 *
	 * @param aLocator the a locator
	 * @return the rest request builder
	 */
	@Override
	default RestRequestBuilder withLocator( String aLocator ) {
		setLocator( aLocator );
		return this;
	}

	/**
	 * With http method.
	 *
	 * @param aHttpMethod the a http method
	 * @return the rest request builder
	 */
	@Override
	default RestRequestBuilder withHttpMethod( HttpMethod aHttpMethod ) {
		setHttpMethod( aHttpMethod );
		return this;
	}

	/**
	 * With request.
	 *
	 * @param <REQ> the generic type
	 * @param aRequest the a request
	 * @return the rest request builder
	 */
	@Override
	default <REQ> RestRequestBuilder withRequest( REQ aRequest ) {
		setRequest( aRequest );
		return this;
	}

	/**
	 * With Header-Fields.
	 *
	 * @param aRequestHeaderFields the a request Header-Fields
	 * @return the rest request builder
	 */
	@Override
	default RestRequestBuilder withHeaderFields( RequestHeaderFields aRequestHeaderFields ) {
		setHeaderFields( aRequestHeaderFields );
		return this;
	}
}
