// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import org.refcodes.net.BaseLocatorAccessor.BaseLocatorBuilder;
import org.refcodes.net.BaseLocatorAccessor.BaseLocatorProperty;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.MediaTypeFactoryLookup.MutableMediaTypeFactoryLookup;
import org.refcodes.net.RealmAccessor.RealmBuilder;
import org.refcodes.net.RealmAccessor.RealmProperty;
import org.refcodes.observer.Observable;
import org.refcodes.runtime.RequestCorrelation;
import org.refcodes.runtime.SessionCorrelation;

/**
 * The {@link RestServer} acts as the target for clients issuing REST requests.
 * {@link RestEndpointBuilder} instances, most easily being created with the
 * {@link #onRequest(HttpMethod, String, RestRequestObserver)} method (or the
 * like), are registered as listeners to the {@link RestServer} (
 * {@link HttpRestServer}). The {@link RestServer} ({@link HttpRestServer})
 * fires {@link RestRequestEvent} events to the {@link RestRequestObserver}s of
 * an {@link RestEndpoint} dedicated to an according locator (pattern) for a
 * specific {@link HttpMethod}.
 */
public interface RestServer extends Observable<RestEndpoint>, MutableMediaTypeFactoryLookup, RealmProperty, RealmBuilder<RestServer>, BaseLocatorProperty, BaseLocatorBuilder<RestServer>, RequestCorrelation<RestServer>, SessionCorrelation<RestServer> {
	/**
	 * With session correlation.
	 *
	 * @param hasSessionCorrelation the has session correlation
	 * @return the rest server
	 */
	@Override
	default RestServer withSessionCorrelation( boolean hasSessionCorrelation ) {
		setSessionCorrelation( hasSessionCorrelation );
		return this;
	}

	/**
	 * With enable session correlation.
	 *
	 * @return the rest server
	 */
	@Override
	default RestServer withEnableSessionCorrelation() {
		enableSessionCorrelation();
		return this;
	}

	/**
	 * With disable session correlation.
	 *
	 * @return the rest server
	 */
	@Override
	default RestServer withDisableSessionCorrelation() {
		disableSessionCorrelation();
		return this;
	}

	/**
	 * With request correlation.
	 *
	 * @param hasRequestCorrelation the has request correlation
	 * @return the rest server
	 */
	@Override
	default RestServer withRequestCorrelation( boolean hasRequestCorrelation ) {
		setRequestCorrelation( hasRequestCorrelation );
		return this;
	}

	/**
	 * With enable request correlation.
	 *
	 * @return the rest server
	 */
	@Override
	default RestServer withEnableRequestCorrelation() {
		enableRequestCorrelation();
		return this;
	}

	/**
	 * With disable request correlation.
	 *
	 * @return the rest server
	 */
	@Override
	default RestServer withDisableRequestCorrelation() {
		disableRequestCorrelation();
		return this;
	}

	/**
	 * For the sake of unified naming, this method just delegates to
	 * {@link #subscribeObserver(Object)}.
	 *
	 * @param aRestEndpoint the a rest endpoint
	 * @return true, if successful
	 */
	default boolean onRequest( RestEndpoint aRestEndpoint ) {
		return subscribeObserver( aRestEndpoint );
	}

	/**
	 * With base locator.
	 *
	 * @param aBaseLocator the a base locator
	 * @return the rest server
	 */
	@Override
	default RestServer withBaseLocator( String aBaseLocator ) {
		setBaseLocator( aBaseLocator );
		return this;
	}

	/**
	 * With realm.
	 *
	 * @param aRealm the a realm
	 * @return the rest server
	 */
	@Override
	default RestServer withRealm( String aRealm ) {
		setRealm( aRealm );
		return this;
	}

	/**
	 * Registers a pre-configured {@link RestEndpoint}, triggerd by any HTTP
	 * method, and returns its instance. To unsubscribe via
	 * {@link #unsubscribeObserver(Object)}, use the returned instance.
	 * 
	 * Attention: Invoke {@link RestEndpointBuilder#open()} to activate this
	 * endpoint!
	 * 
	 * @param aLocatorPattern The local Locator-Pattern to which this
	 *        {@link RestEndpoint} is bound.
	 * @param aRequestObserver The listener processing a request targeted at
	 *        this {@link RestEndpoint}.
	 * 
	 * @return The pre-configured {@link RestEndpoint} which acts as handle to
	 *         unsubscribe the {@link RestEndpoint} via
	 *         {@link #unsubscribeObserver(Object)}.
	 */
	default RestEndpointBuilder onRequest( String aLocatorPattern, RestRequestObserver aRequestObserver ) {
		return onRequest( null, aLocatorPattern, aRequestObserver );
	}

	/**
	 * Registers a pre-configured {@link RestEndpoint} with the least required
	 * attributes and returns its instance. To unsubscribe via
	 * {@link #unsubscribeObserver(Object)}, use the returned instance.
	 * 
	 * Attention: Invoke {@link RestEndpointBuilder#open()} to activate this
	 * endpoint!
	 * 
	 * @param aHttpMethod The HTTP-Method to which this {@link RestEndpoint} is
	 *        bound. A value of <code>null</code> means that any HTTP-Method may
	 *        trigger this {@link RestEndpoint}.
	 * @param aLocatorPattern The local Locator-Pattern to which this
	 *        {@link RestEndpoint} is bound. A value of <code>null</code> means
	 *        that any locator may trigger this {@link RestEndpoint}.
	 * @param aRequestObserver The listener processing a request targeted at
	 *        this {@link RestEndpoint}.
	 * 
	 * @return The pre-configured {@link RestEndpoint} which acts as handle to
	 *         unsubscribe the {@link RestEndpoint} via
	 *         {@link #unsubscribeObserver(Object)}.
	 */
	RestEndpointBuilder onRequest( HttpMethod aHttpMethod, String aLocatorPattern, RestRequestObserver aRequestObserver );

	/**
	 * Even more convenient: Everything done here can also be done using
	 * {@link #onRequest(HttpMethod, String, RestRequestObserver)}:
	 * 
	 * Registers a {@link RestRequestObserver} with the given Locator-Pattern to
	 * incoming GET methods.
	 * 
	 * Attention: Invoke {@link RestEndpointBuilder#open()} to activate this
	 * endpoint!
	 * 
	 * @param aLocatorPattern The local Locator-Pattern to which this
	 *        {@link RestEndpoint} is bound.
	 * 
	 * @param aRequestObserver The listener processing a request targeted at
	 *        this {@link RestEndpoint}.
	 * 
	 * @return The pre-configured {@link RestEndpoint} which acts as handle to
	 *         unsubscribe the {@link RestEndpoint} via
	 *         {@link #unsubscribeObserver(Object)}.
	 */
	default RestEndpointBuilder onGet( String aLocatorPattern, RestRequestObserver aRequestObserver ) {
		return onRequest( HttpMethod.GET, aLocatorPattern, aRequestObserver );
	}

	/**
	 * Even more convenient: Everything done here can also be done using
	 * {@link #onRequest(HttpMethod, String, RestRequestObserver)}:
	 * 
	 * Registers a {@link RestRequestObserver} with the given Locator-Pattern to
	 * incoming PUT methods.
	 * 
	 * Attention: Invoke {@link RestEndpointBuilder#open()} to activate this
	 * endpoint!
	 * 
	 * @param aLocatorPattern The local Locator-Pattern to which this
	 *        {@link RestEndpoint} is bound.
	 * 
	 * @param aRequestObserver The listener processing a request targeted at
	 *        this {@link RestEndpoint}.
	 * 
	 * @return The pre-configured {@link RestEndpoint} which acts as handle to
	 *         unsubscribe the {@link RestEndpoint} via
	 *         {@link #unsubscribeObserver(Object)}.
	 */
	default RestEndpointBuilder onPut( String aLocatorPattern, RestRequestObserver aRequestObserver ) {
		return onRequest( HttpMethod.PUT, aLocatorPattern, aRequestObserver );
	}

	/**
	 * Even more convenient: Everything done here can also be done using
	 * {@link #onRequest(HttpMethod, String, RestRequestObserver)}:
	 * 
	 * Registers a {@link RestRequestObserver} with the given Locator-Pattern to
	 * incoming POST methods.
	 * 
	 * Attention: Invoke {@link RestEndpointBuilder#open()} to activate this
	 * endpoint!
	 * 
	 * @param aLocatorPattern The local Locator-Pattern to which this
	 *        {@link RestEndpoint} is bound.
	 * 
	 * @param aRequestObserver The listener processing a request targeted at
	 *        this {@link RestEndpoint}.
	 * 
	 * @return The pre-configured {@link RestEndpoint} which acts as handle to
	 *         unsubscribe the {@link RestEndpoint} via
	 *         {@link #unsubscribeObserver(Object)}.
	 */
	default RestEndpointBuilder onPost( String aLocatorPattern, RestRequestObserver aRequestObserver ) {
		return onRequest( HttpMethod.POST, aLocatorPattern, aRequestObserver );
	}

	/**
	 * Even more convenient: Everything done here can also be done using
	 * {@link #onRequest(HttpMethod, String, RestRequestObserver)}:
	 * 
	 * Registers a {@link RestRequestObserver} with the given Locator-Pattern to
	 * incoming DELETE methods.
	 * 
	 * Attention: Invoke {@link RestEndpointBuilder#open()} to activate this
	 * endpoint!
	 * 
	 * @param aLocatorPattern The local Locator-Pattern to which this
	 *        {@link RestEndpoint} is bound.
	 * 
	 * @param aRequestObserver The listener processing a request targeted at
	 *        this {@link RestEndpoint}.
	 * 
	 * @return The pre-configured {@link RestEndpoint} which acts as handle to
	 *         unsubscribe the {@link RestEndpoint} via
	 *         {@link #unsubscribeObserver(Object)}.
	 */
	default RestEndpointBuilder onDelete( String aLocatorPattern, RestRequestObserver aRequestObserver ) {
		return onRequest( HttpMethod.DELETE, aLocatorPattern, aRequestObserver );
	}
}
