// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest.impls;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetSocketAddress;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.Base64;
import java.util.concurrent.ExecutorService;

import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLParameters;
import javax.net.ssl.TrustManagerFactory;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatus;
import org.refcodes.component.OpenException;
import org.refcodes.data.Delimiter;
import org.refcodes.data.LatencySleepTime;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.impls.RuntimeLoggerFactorySingleton;
import org.refcodes.net.AuthType;
import org.refcodes.net.BadResponseException;
import org.refcodes.net.BasicAuthCredentials;
import org.refcodes.net.BasicAuthObserver;
import org.refcodes.net.BasicAuthRequiredException;
import org.refcodes.net.BasicAuthResponse;
import org.refcodes.net.HeaderField;
import org.refcodes.net.HeaderFields;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpServerResponse;
import org.refcodes.net.HttpStatusCode;
import org.refcodes.net.HttpStatusException;
import org.refcodes.net.HttpsConnectionRequestObserver;
import org.refcodes.net.MediaTypeFactory;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.impls.ApplicationFormFactory;
import org.refcodes.net.impls.ApplicationJsonFactory;
import org.refcodes.net.impls.ApplicationXmlFactory;
import org.refcodes.net.impls.RequestHeaderFieldsImpl;
import org.refcodes.net.impls.TextPlainFactory;
import org.refcodes.rest.HttpRestServer;
import org.refcodes.rest.RestEndpoint;
import org.refcodes.rest.RestRequestObserver;

import com.sun.net.httpserver.Authenticator;
import com.sun.net.httpserver.Headers;
import com.sun.net.httpserver.HttpContext;
import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpPrincipal;
import com.sun.net.httpserver.HttpServer;
import com.sun.net.httpserver.HttpsConfigurator;
import com.sun.net.httpserver.HttpsParameters;
import com.sun.net.httpserver.HttpsServer;

/**
 * Implementation of the {@link HttpRestServer} interface using the
 * {@link HttpServer} defined in the <code>com.sun.net.httpserver</code>
 * package.
 * 
 * The {@link HttpRestServerImpl} is being initialized with some common
 * {@link MediaTypeFactory} instances (as implemented by the
 * {@link AbstractRestServer}). At the time of writing this document the
 * {@link MediaTypeFactory} instances being pre-configured are:
 * 
 * <ul>
 * <li>{@link ApplicationJsonFactory}</li>
 * <li>{@link ApplicationXmlFactory}</li>
 * <li>{@link TextPlainFactory}</li>
 * <li>{@link ApplicationFormFactory}</li>
 * </ul>
 * 
 * The {@link HttpRestServerImpl} supports HTTP as well as HTTPS protocols as
 * being based on the {@link HttpServer} as well as on the {@link HttpsServer}.
 * 
 * For opening up an HTTPS connection, refer to the methods such as
 * {@link #open(File, String, Integer)} or
 * {@link #open(String, File, String, String, String, Integer)} and the like.
 */
public class HttpRestServerImpl extends AbstractRestServer implements HttpRestServer {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	protected static final String CONTEXT_PATH = Delimiter.PATH.getChar() + "";

	private static final int NO_RESPONSE_BODY = -1;

	private static final int CHUNCKED_ENCODING = 0;

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private HttpServer _httpServer = null;

	private HttpsConnectionRequestObserver _httpsConnectionRequestObserver = null;

	private ExecutorService _executorService;

	private ConnectionStatus _connectionStatus = ConnectionStatus.NONE;

	private HttpBasicAuthenticator _httpBasicAuthenticator = null;

	private HttpContext _httpContext;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link HttpRestServerImpl} listening to the given
	 * localhost's port for HTTP-Requests.
	 */
	public HttpRestServerImpl() {
		this( null );
	}

	/**
	 * Constructs a {@link HttpRestServerImpl} listening to the given
	 * localhost's port for HTTP-Requests.
	 *
	 * @param aExecutorService An executor service to be used when creating
	 *        {@link Thread}s.
	 */
	public HttpRestServerImpl( ExecutorService aExecutorService ) {
		super( aExecutorService );
		_executorService = aExecutorService;
	}

	// /////////////////////////////////////////////////////////////////////////
	// LIFECYCLE:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public void open( Integer aPort ) throws OpenException {
		open( aPort, -1 );
	}

	@Override
	public synchronized void open( Integer aPort, int aMaxConnections ) throws OpenException {
		try {
			HttpServer theHttpServer = HttpServer.create();
			theHttpServer.bind( new InetSocketAddress( aPort ), aMaxConnections );
			open( theHttpServer );
		}
		catch ( IOException e ) {
			throw new OpenException( "Unable to bind to port <" + aPort + ">: " + ExceptionUtility.toMessage( e ), e );
		}
	}

	@Override
	public synchronized void close() throws CloseException {
		if ( _connectionStatus != ConnectionStatus.OPENED ) {
			throw new CloseException( "Connection is in status <" + _connectionStatus + ">. Open the connection before closing!" );
		}
		try {
			if ( _httpServer != null ) {
				_httpServer.stop( LatencySleepTime.MIN.getMilliseconds() / 1000 );
				_httpServer.removeContext( CONTEXT_PATH );
				_httpServer = null;
			}
		}
		finally {
			_connectionStatus = ConnectionStatus.CLOSED;
		}
	}

	@Override
	public void open( String aSecureSocketProtocol, File aKeyStoreFile, String aKeyStoreType, String aKeyStorePassword, String aKeyPassword, Integer aPort, int aMaxConnections ) throws OpenException {
		try {
			InetSocketAddress theAddress = new InetSocketAddress( aPort );
			SSLContext theSSLContext = SSLContext.getInstance( aSecureSocketProtocol ); // "TLS"
			KeyStore theKeyStore = KeyStore.getInstance( aKeyStoreType ); // "JKS"
			FileInputStream theKeystoreInputStream = new FileInputStream( aKeyStoreFile );
			theKeyStore.load( theKeystoreInputStream, aKeyStorePassword.toCharArray() );
			KeyManagerFactory theKeyManagerFactory = KeyManagerFactory.getInstance( KeyManagerFactory.getDefaultAlgorithm() );
			theKeyManagerFactory.init( theKeyStore, aKeyPassword.toCharArray() );
			TrustManagerFactory theTrustManagerFactory = TrustManagerFactory.getInstance( TrustManagerFactory.getDefaultAlgorithm() );
			theTrustManagerFactory.init( theKeyStore );
			theSSLContext.init( theKeyManagerFactory.getKeyManagers(), theTrustManagerFactory.getTrustManagers(), null );
			HttpsServer theHttpsServer = HttpsServer.create( theAddress, aMaxConnections );
			theHttpsServer.setHttpsConfigurator( new HttpsRestConfigurator( theSSLContext ) );
			theHttpsServer.setExecutor( null ); // creates a default executor
			open( theHttpsServer );
		}
		catch ( IOException | NoSuchAlgorithmException | KeyStoreException | CertificateException | UnrecoverableKeyException | KeyManagementException e ) {
			throw new OpenException( "Unable to bind to port <" + aPort + ">: " + ExceptionUtility.toMessage( e ), e );
		}
	}

	@Override
	public HttpRestServer onConnectionRequest( HttpsConnectionRequestObserver aObserver ) {
		_httpsConnectionRequestObserver = aObserver;
		return this;
	}

	@Override
	public HttpRestServer onBasicAuthRequest( BasicAuthObserver aBasicAuthObserver ) {
		HttpContext theHttpContext = _httpContext;
		HttpBasicAuthenticator theHttpBasicAuthenticator = null;
		if ( aBasicAuthObserver != null ) {
			theHttpBasicAuthenticator = new HttpBasicAuthenticator( aBasicAuthObserver );
			if ( theHttpContext != null ) {
				theHttpContext.setAuthenticator( theHttpBasicAuthenticator );
			}
		}
		_httpBasicAuthenticator = theHttpBasicAuthenticator;
		return this;
	}

	@Override
	public HttpRestServer withRealm( String aRealm ) {
		setRealm( aRealm );
		return this;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Gets the http server.
	 *
	 * @return the http server
	 */
	protected HttpServer getHttpServer() {
		return _httpServer;
	}

	/**
	 * A hook to be used when using custom {@link HttpServer} (
	 * {@link HttpsServer}) by custom open(...) methods of sub-classes of this
	 * {@link HttpRestServerImpl}. E.g {@link HttpRestServerImpl} uses this hook
	 * to pre-configure a {@link HttpsServer} for HTTPS.
	 * 
	 * The passed {@link HttpServer} ( {@link HttpsServer}) must already be
	 * bound to a port and enabled with the number of concurrent connections as
	 * of {@link HttpServer#bind(InetSocketAddress, int)}.
	 * 
	 * @param aHttpServer The {@link HttpServer} to be used. E.g. an
	 *        {@link HttpsServer} might be used to enable HTTPS.
	 * 
	 * @throws IOException in case opening with the provided
	 *         {@link HttpRestServer} fails.
	 */
	protected void open( HttpServer aHttpServer ) throws IOException {
		if ( _connectionStatus == ConnectionStatus.OPENED ) {
			throw new OpenException( "Connection is still in status <" + _connectionStatus + ">. Close the connection before reopening!" );
		}
		if ( _executorService != null ) aHttpServer.setExecutor( _executorService );
		HttpContext theHttpContext = aHttpServer.createContext( CONTEXT_PATH, new EndpointHttpHandler() );

		HttpBasicAuthenticator theHttpBasicAuthenticator = _httpBasicAuthenticator;
		if ( theHttpBasicAuthenticator != null ) {
			theHttpContext.setAuthenticator( theHttpBasicAuthenticator );
		}
		aHttpServer.start();
		_httpServer = aHttpServer;
		_httpContext = theHttpContext;
		_connectionStatus = ConnectionStatus.OPENED;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Handles the {@link Result} in case of authorization failure.
	 * 
	 * @param aHttpExchange The {@link com.sun.net.httpserver.HttpExchange}
	 *        which's "WWW-Authenticate" header is to be modified.
	 * 
	 * @return the {@link Result} of type {@link Authenticator.Failure}.
	 */
	private Authenticator.Result toBasicAuthFailure( HttpExchange aHttpExchange ) {
		return new Authenticator.Failure( HttpStatusCode.UNAUTHORIZED.getStatusCode() );
	}

	/**
	 * Handles the {@link Result} in case of authorization required.
	 * 
	 * @param aHttpExchange The {@link com.sun.net.httpserver.HttpExchange}
	 *        which's "WWW-Authenticate" header is to be modified.
	 * 
	 * @return the {@link Result} of type {@link Retry}.
	 */
	private Authenticator.Result toBasicAuthRequired( HttpExchange aHttpExchange ) {
		Headers theHeaders = aHttpExchange.getResponseHeaders();
		setBasicAuthRequired( theHeaders );
		return new Authenticator.Retry( HttpStatusCode.UNAUTHORIZED.getStatusCode() );
	}

	/**
	 * Sets Basic-Authentication required for the
	 * {@link HeaderField#WWW_AUTHENTICATE}.
	 * 
	 * @param aHeaders The headers which to modify accordingly.
	 */
	private void setBasicAuthRequired( Headers aHeaders ) {
		aHeaders.set( HeaderField.WWW_AUTHENTICATE.getName(), HeaderFields.BASIC_REALM + "=\"" + getRealm() + "\"" );
	}

	///////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	///////////////////////////////////////////////////////////////////////////

	/**
	 * Extension of the {@link com.sun.net.httpserver.Authenticator} for doing
	 * custom HTTP Basic-Authentication.
	 */
	private class HttpBasicAuthenticator extends Authenticator {

		private BasicAuthObserver _basicAuthObserver;

		/**
		 * Instantiates a new http basic authenticator.
		 *
		 * @param aBasicAuthObserver the a basic auth observer
		 */
		public HttpBasicAuthenticator( BasicAuthObserver aBasicAuthObserver ) {
			_basicAuthObserver = aBasicAuthObserver;
		}

		/**
		 * Authenticate.
		 *
		 * @param aHttpExchange the a http exchange
		 * @return the result
		 */
		@Override
		public Result authenticate( HttpExchange aHttpExchange ) {
			Headers theRequestHeaders = (Headers) aHttpExchange.getRequestHeaders();
			String theAuth = theRequestHeaders.getFirst( HeaderField.AUTHORIZATION.getName() );
			BasicAuthCredentials theCredentials = null;
			if ( theAuth == null ) {
				return toBasicAuthRequired( aHttpExchange );
			}
			else {
				int theMarker = theAuth.indexOf( ' ' );
				if ( theMarker == -1 || !theAuth.substring( 0, theMarker ).equals( AuthType.BASIC.getName() ) ) {
					return toBasicAuthFailure( aHttpExchange );
				}
				byte[] theCredentialChars = Base64.getDecoder().decode( theAuth.substring( theMarker + 1 ) );
				String theCredentialsText = new String( theCredentialChars );
				theMarker = theCredentialsText.indexOf( ':' );
				theCredentials = BasicAuthCredentials.toBasicAuthCredentials( theCredentialsText.substring( 0, theMarker ), theCredentialsText.substring( theMarker + 1 ) );
			}

			BasicAuthResponse theBasicAuthResponse = _basicAuthObserver.onBasicAuthRequest( aHttpExchange.getLocalAddress(), aHttpExchange.getRemoteAddress(), HttpMethod.fromHttpMethod( aHttpExchange.getRequestMethod() ), aHttpExchange.getRequestURI().getPath(), theCredentials, getRealm() );
			if ( theBasicAuthResponse == null ) {
				throw new NullPointerException( "Your <HttpBasicAuthenticator> instance must return an element of type <BasicAuthResponse> and not null." );
			}
			switch ( theBasicAuthResponse ) {
			case BASIC_AUTH_SUCCESS:
				return new Authenticator.Success( new HttpPrincipal( theCredentials.getUserName(), getRealm() ) );
			default:
			case BASIC_AUTH_FAILURE:
				return toBasicAuthFailure( aHttpExchange );
			}
		}
	}

	/**
	 * Main {@link com.sun.net.httpserver.HttpHandler} managing the dispatch of
	 * incoming requests to the registered {@link RestRequestObserver} instances
	 * depending on the according {@link RestEndpoint}'s Locator-Pattern, HTTP
	 * method and so on.
	 */
	private class EndpointHttpHandler implements HttpHandler {

		/**
		 * Handle.
		 *
		 * @param aHttpExchange the a http exchange
		 * @throws IOException Signals that an I/O exception has occurred.
		 */
		@Override
		public void handle( HttpExchange aHttpExchange ) throws IOException {
			HttpMethod theHttpMethod = HttpMethod.fromHttpMethod( aHttpExchange.getRequestMethod() );
			if ( theHttpMethod != null ) {
				try {
					InetSocketAddress theLocalAddress = aHttpExchange.getLocalAddress();
					InetSocketAddress theRemoteAddress = aHttpExchange.getRemoteAddress();

					// Decompose the request -->
					Headers theRequestHeaders = aHttpExchange.getRequestHeaders();
					RequestHeaderFields theRequestHeaderFields = new RequestHeaderFieldsImpl( theRequestHeaders );
					String theLocator = aHttpExchange.getRequestURI().getPath();
					String theQueryString = aHttpExchange.getRequestURI().getQuery();
					// <-- Decompose the request

					// Dispatch the request-->
					HttpServerResponse theServerResponse = onHttpRequest( theLocalAddress, theRemoteAddress, theHttpMethod, theLocator, theQueryString, theRequestHeaderFields, aHttpExchange.getRequestBody() );
					// <-- Dispatch the request
					// Assemble the response headers to be sent -->
					Headers theResponseHeaders = aHttpExchange.getResponseHeaders();
					for ( String eKey : theServerResponse.getHeaderFields().keySet() ) {
						theResponseHeaders.put( eKey, theServerResponse.getHeaderFields().get( eKey ) );
					}
					// <-- Assemble the response headers to be sent

					// Send the response -->
					Integer theStatusCode = HttpStatusCode.OK.getStatusCode();
					HttpStatusCode theResponsesCode = theServerResponse.getHttpStatusCode();
					if ( theResponsesCode != null ) theStatusCode = theResponsesCode.getStatusCode();

					// Marshal response as of the response CONTENT and request's
					// ACCEPT header and request's CONTENT -->
					// |--> Process the response:
					Object theResponse = theServerResponse.getResponse();
					// |--> Input-Stream:
					if ( (theResponse instanceof InputStream) ) {
						aHttpExchange.sendResponseHeaders( theStatusCode, CHUNCKED_ENCODING );
						InputStream theInputStream = (InputStream) theResponse;
						HttpRestClientImpl.pipe( theInputStream, aHttpExchange.getResponseBody() );
						aHttpExchange.getResponseBody().flush();
					}
					// Input-Stream <--|
					// Object |-->
					else {
						byte[] theBytes = null;
						if ( theResponse != null ) {
							try {
								theBytes = theServerResponse.toHttpBody().getBytes();
							}
							catch ( BadResponseException e ) {
								LOGGER.warn( e.getMessage() + " Trying fallback procedure ..." );
								theBytes = toResponseBody( theServerResponse.getResponse(), theRequestHeaderFields, theServerResponse.getHeaderFields() );
							}
						}
						if ( theBytes != null && theBytes.length != 0 ) {
							aHttpExchange.sendResponseHeaders( theStatusCode, theBytes.length );
							aHttpExchange.getResponseBody().write( theBytes );
							aHttpExchange.getResponseBody().flush();
						}
						else {
							aHttpExchange.sendResponseHeaders( theStatusCode, NO_RESPONSE_BODY );
						}
					}
					// <-- Marshal response as of the response CONTENT and
					// request's ACCEPT header and request's CONTENT
					aHttpExchange.getResponseBody().close();
					// <-- Send the response
				}
				catch ( BasicAuthRequiredException e ) {
					setBasicAuthRequired( aHttpExchange.getResponseHeaders() );
					LOGGER.info( "Required HTTP Basic-Authentication with status <" + e.getStatusCode() + "> with code <" + e.getStatusCode().getStatusCode() + "> for request URL <" + aHttpExchange.getRequestURI() + "> with request method <" + aHttpExchange.getRequestMethod() + ">: " + ExceptionUtility.toMessage( e ) );
					aHttpExchange.sendResponseHeaders( e.getStatusCode().getStatusCode(), NO_RESPONSE_BODY );
					aHttpExchange.getResponseBody().close();
				}
				catch ( HttpStatusException e ) {
					LOGGER.warn( "Responding status <" + e.getStatusCode() + "> with code <" + e.getStatusCode().getStatusCode() + "> for request URL <" + aHttpExchange.getRequestURI() + "> with request method <" + aHttpExchange.getRequestMethod() + ">: " + ExceptionUtility.toMessage( e ) );
					aHttpExchange.sendResponseHeaders( e.getStatusCode().getStatusCode(), NO_RESPONSE_BODY );
					aHttpExchange.getResponseBody().close();
				}
				catch ( Exception e ) {
					LOGGER.warn( "Bad request <" + e.getClass().getName() + "> for request URL <" + aHttpExchange.getRequestURI() + "> with request method <" + aHttpExchange.getRequestMethod() + ">: " + ExceptionUtility.toMessage( e ), e );
					aHttpExchange.sendResponseHeaders( HttpStatusCode.BAD_REQUEST.getStatusCode(), NO_RESPONSE_BODY );
					aHttpExchange.getResponseBody().close();
				}
			}
			else {
				LOGGER.warn( "Unknown HTTP-Method <" + aHttpExchange.getRequestMethod() + "> when querying resource locator <" + aHttpExchange.getLocalAddress() + ">." );
				aHttpExchange.sendResponseHeaders( HttpStatusCode.INTERNAL_SERVER_ERROR.getStatusCode(), NO_RESPONSE_BODY );
			}
		}
	}

	/**
	 * This class is used to configure the HTTPS parameters for each incoming
	 * HTTPS connection in order to change the default configuration:.
	 */
	private class HttpsRestConfigurator extends HttpsConfigurator {

		/**
		 * {@inheritDoc}
		 */
		public HttpsRestConfigurator( SSLContext aSSLContext ) {
			super( aSSLContext );
		}

		/**
		 * TODO: Not sure on how to proceed with
		 * {@link HttpsParameters#setNeedClientAuth(boolean)}. Do we need to
		 * propagate such a flag back from
		 * {@link HttpsConnectionRequestObserver} to the
		 * {@link HttpsParameters#setNeedClientAuth(boolean)}?
		 *
		 * @param theHttpsParams the the https params
		 */
		@Override
		public void configure( HttpsParameters theHttpsParams ) {

			HttpsConnectionRequestObserver theObserver = _httpsConnectionRequestObserver;
			if ( theObserver != null ) {

				InetSocketAddress theRemoteAddress = theHttpsParams.getClientAddress();
				InetSocketAddress theLocalAddress = null;
				HttpServer theServer = getHttpServer();
				if ( theServer != null ) {
					theLocalAddress = theServer.getAddress();
				}
				else {
					LOGGER.warn( "Unable to determine the local address for remote address <" + theRemoteAddress.toString() + ">, the server might have been closed in the meantimne." );
				}
				SSLContext theSSLContext = getSSLContext();
				SSLParameters theSSLParams = theSSLContext.getDefaultSSLParameters();
				theObserver.onHttpsConnectionRequest( theLocalAddress, theRemoteAddress, theSSLParams );
				theHttpsParams.setSSLParameters( theSSLParams );
			}
		}
	}
}
