// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest.impls;

import static org.junit.Assert.*;
import static org.refcodes.rest.HttpRestClientSugar.*;
import static org.refcodes.rest.HttpRestServerSugar.*;
import static org.refcodes.rest.HttpRestServerSugar.open;

import java.io.IOException;

import org.junit.Test;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.impls.RuntimeLoggerFactorySingleton;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.impls.PortManagerSingleton;
import org.refcodes.rest.RestEndpointBuilder;
import org.refcodes.rest.RestResponse;

/**
 * The Class HttpRestSugarTest.
 */
public class HttpRestSugarTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String BASE_URL = "http://localhost";

	private static final String BASE_LOCATOR = "/refcodes";

	private static final String LOCATOR = "/bla";

	private static final String LAST_NAME = "Bushnell";

	private static final String FIRST_NAME = "Nolan";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Test asynchronous rest sugar.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testAsynchronousRestSugar() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		withBaseLocator( BASE_LOCATOR );
		RestEndpointBuilder theObserver = onPost( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} );
		theObserver.open();
		open( thePort );

		withBaseUrl( toBaseUrl( thePort ) );
		doPost( LOCATOR, ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			LOGGER.debug( thePerson.toString() );
			closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new Person( FIRST_NAME, LAST_NAME ) ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}
		unsubscribeObserver( theObserver );
		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test synchronous rest sugar.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws HttpResponseException the http response exception
	 */
	@Test
	public void testSynchronousRestSugar() throws IOException, HttpResponseException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		withBaseLocator( BASE_LOCATOR );
		RestEndpointBuilder theObserver = onPost( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} );
		theObserver.open();
		open( thePort );

		withBaseUrl( toBaseUrl( thePort ) );
		RestResponse theResponse = doPost( LOCATOR, new Person( FIRST_NAME, LAST_NAME ) );

		Person thePerson = theResponse.getResponse( Person.class );
		assertEquals( FIRST_NAME, thePerson.getFirstName() );
		assertEquals( LAST_NAME, thePerson.getLastName() );
		LOGGER.debug( thePerson.toString() );

		unsubscribeObserver( theObserver );
		closeQuietly();
		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * To base url.
	 *
	 * @param thePort the the port
	 * @return the string
	 */
	private String toBaseUrl( Integer thePort ) {
		return BASE_URL + ":" + thePort + BASE_LOCATOR;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Class Person.
	 */
	public static class Person {

		/**
		 * To string.
		 *
		 * @return the string
		 */
		@Override
		public String toString() {
			return "Person [firstName=" + firstName + ", lastName=" + lastName + "]";
		}

		private String firstName;
		private String lastName;

		/**
		 * Instantiates a new person.
		 */
		public Person() {};

		/**
		 * Instantiates a new person.
		 *
		 * @param firstName the first name
		 * @param lastName the last name
		 */
		public Person( String firstName, String lastName ) {
			this.firstName = firstName;
			this.lastName = lastName;
		}

		/**
		 * Gets the first name.
		 *
		 * @return the first name
		 */
		public String getFirstName() {
			return firstName;
		}

		/**
		 * Sets the first name.
		 *
		 * @param firstName the new first name
		 */
		public void setFirstName( String firstName ) {
			this.firstName = firstName;
		}

		/**
		 * Gets the last name.
		 *
		 * @return the last name
		 */
		public String getLastName() {
			return lastName;
		}

		/**
		 * Sets the last name.
		 *
		 * @param lastName the new last name
		 */
		public void setLastName( String lastName ) {
			this.lastName = lastName;
		}
	}
}
