// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest.impls;

import static org.junit.Assert.*;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;

import org.junit.Ignore;
import org.junit.Test;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.impls.RuntimeLoggerFactorySingleton;
import org.refcodes.net.HttpMethod;
import org.refcodes.rest.HttpRestClient;
import org.refcodes.rest.HttpRestServer;

/**
 * The Class HttpsRestServerTest.
 */
public class HttpsRestServerTest {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String LOCATOR = "/bla";

	private static final String BASE_URL = "https://localhost:5161/refcodes";

	private static final String BASE_LOCATOR = "/refcodes";

	private static final String LAST_NAME = "Bushnell";

	private static final String FIRST_NAME = "Nolan";

	private static final String KEY_STORE_A = "keystore_a.jks";

	private static final String KEY_STORE_A_PASSWORD = "keyStoreSecret";

	private static final String KEY_A_PASSWORD = "keySecret";

	private static final String KEY_STORE_B = "keystore_b.jks";

	private static final String KEY_STORE_B_PASSWORD = "keyStoreSecret";

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private boolean _hasConnectionRequest;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * "keytool -genkey -keyalg RSA -alias localhost -keystore keystore_a.jks
	 * -storepass keyStoreSecret -keypass keySecret"
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws URISyntaxException the URI syntax exception
	 */
	@Test
	// Ignore("Either the one or the other test can be executed in a JVM
	// instance.")
	public void testHttpsRestServerA() throws IOException, URISyntaxException {

		_hasConnectionRequest = false;
		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} ).open();
		theRestServer.onConnectionRequest( ( aLocalAddress, aRemoteAddress, aHttpsParams ) -> {
			LOGGER.info( "Local := " + aLocalAddress + ", remote := " + aRemoteAddress + ", HTTP params := " + aHttpsParams );
			if ( _hasConnectionRequest ) {
				fail( "Expecting just one connection request." );
			}
			_hasConnectionRequest = true;
		} );
		theRestServer.open( toResourceFile( KEY_STORE_A ), KEY_STORE_A_PASSWORD, KEY_A_PASSWORD, 5161 );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( BASE_URL );
		theRestClient.open( toResourceFile( KEY_STORE_A ), KEY_STORE_A_PASSWORD );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			LOGGER.debug( thePerson.toString() );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new Person( FIRST_NAME, LAST_NAME ) ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}
		assertTrue( _hasConnectionRequest );
	}

	/**
	 * "keytool -genkey -keyalg RSA -alias localhost -keystore keystore_b.jks
	 * -storepass keyStoreSecret"
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws URISyntaxException the URI syntax exception
	 */
	@Test
	@Ignore("Either the one or the other test can be executed in a JVM instance.")
	public void testHttpsRestServerB() throws IOException, URISyntaxException {

		_hasConnectionRequest = false;
		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} ).open();
		theRestServer.onConnectionRequest( ( aLocalAddress, aRemoteAddress, aHttpsParams ) -> {
			LOGGER.info( "Local := " + aLocalAddress + ", remote := " + aRemoteAddress + ", HTTP params := " + aHttpsParams );
			if ( _hasConnectionRequest ) {
				fail( "Expecting just one connection request." );
			}
			_hasConnectionRequest = true;
		} );
		theRestServer.open( toResourceFile( KEY_STORE_B ), KEY_STORE_B_PASSWORD, 5161 );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( BASE_URL );
		theRestClient.open( toResourceFile( KEY_STORE_B ), KEY_STORE_B_PASSWORD );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			LOGGER.debug( thePerson.toString() );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new Person( FIRST_NAME, LAST_NAME ) ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}
		assertTrue( _hasConnectionRequest );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * To resource file.
	 *
	 * @param aKeystoreFile the a keystore file
	 * @return the file
	 * @throws URISyntaxException the URI syntax exception
	 */
	protected File toResourceFile( String aKeystoreFile ) throws URISyntaxException {
		return new File( HttpsRestServerTest.class.getResource( "/" + aKeystoreFile ).toURI() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Class Person.
	 */
	public static class Person {

		/**
		 * To string.
		 *
		 * @return the string
		 */
		@Override
		public String toString() {
			return "Person [firstName=" + firstName + ", lastName=" + lastName + "]";
		}

		private String firstName;
		private String lastName;

		/**
		 * Instantiates a new person.
		 */
		public Person() {};

		/**
		 * Instantiates a new person.
		 *
		 * @param firstName the first name
		 * @param lastName the last name
		 */
		public Person( String firstName, String lastName ) {
			this.firstName = firstName;
			this.lastName = lastName;
		}

		/**
		 * Gets the first name.
		 *
		 * @return the first name
		 */
		public String getFirstName() {
			return firstName;
		}

		/**
		 * Sets the first name.
		 *
		 * @param firstName the new first name
		 */
		public void setFirstName( String firstName ) {
			this.firstName = firstName;
		}

		/**
		 * Gets the last name.
		 *
		 * @return the last name
		 */
		public String getLastName() {
			return lastName;
		}

		/**
		 * Sets the last name.
		 *
		 * @param lastName the new last name
		 */
		public void setLastName( String lastName ) {
			this.lastName = lastName;
		}
	}
}
