package org.refcodes.rest;

import java.security.KeyStore;

import org.refcodes.component.ConnectionComponent;
import org.refcodes.component.ConnectionStatusAccessor;
import org.refcodes.component.LinkComponent;
import org.refcodes.component.OpenException;
import org.refcodes.net.BaseUrlAccessor.BaseUrlBuilder;
import org.refcodes.net.BaseUrlAccessor.BaseUrlProperty;
import org.refcodes.net.HttpClientContext;
import org.refcodes.net.Url;
import org.refcodes.security.StoreType;
import org.refcodes.security.TrustStoreDescriptor;
import org.refcodes.security.TrustStoreDescriptorAccessor.TrustStoreDescriptorBuilder;
import org.refcodes.security.TrustStoreDescriptorAccessor.TrustStoreDescriptorProperty;

/**
 * Extends a {@link RestClient} to be capable of providing a User-Agent with
 * {@link #setUserAgent(String)} ({@link #withUserAgent(String)}) and to be
 * capable of using base URLs to be set with {@link #setBaseUrl(String)} (
 * {@link #withBaseUrl(String)}).
 * 
 * This type is intended to be used by different separate hierarchy branches by
 * providing the generic type &lt;B&gt;, ensuring a coherent type hierarchy for
 * each branch.
 * 
 * To prepare HTTPS connections, use the methods such as:
 * 
 * {@link #open(Url, TrustStoreDescriptor)} or {@link #open(HttpClientContext)}.
 * 
 * A {@link HttpRestClient} can be shutdown via {@link #close()}.
 */
public interface HttpRestClient extends ConnectionStatusAccessor, ConnectionComponent<HttpClientContext>, RestClient, LinkComponent, BaseUrlProperty, BaseUrlBuilder<HttpRestClient>, TrustStoreDescriptorProperty, TrustStoreDescriptorBuilder<HttpRestClient> {

	String DEFAULT_SSL_PROTOCOL = "TLS";

	String DEFAULT_KEYSTORE_TYPE = StoreType.JKS.name();

	/**
	 * {@inheritDoc}
	 */
	@Override
	default void open() throws OpenException {
		open( getBaseUrl(), getTrustStoreDescriptor() );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default void open( HttpClientContext aCtx ) throws OpenException {
		open( aCtx.getBaseUrl(), aCtx.getTrustStoreDescriptor() );
	}

	/**
	 * Configures the HTTPS client connection with the provided configuration
	 * parameters.
	 * 
	 * @param aBaseUrl The base {@link Url} to be used.
	 * 
	 * @throws OpenException thrown in case something went wrong.
	 */
	default void open( Url aBaseUrl ) throws OpenException {
		open( aBaseUrl, getTrustStoreDescriptor() );
	}

	/**
	 * Configures the HTTPS client connection with the provided configuration
	 * parameters.
	 * 
	 * @param aStoreDescriptor The {@link TrustStoreDescriptor} pointing to your
	 *        {@link KeyStore}.
	 * 
	 * @throws OpenException thrown in case something went wrong.
	 */
	default void open( TrustStoreDescriptor aStoreDescriptor ) throws OpenException {
		open( getBaseUrl(), aStoreDescriptor );
	}

	/**
	 * Configures the HTTPS client connection with the provided configuration
	 * parameters.
	 * 
	 * @param aBaseUrl The base {@link Url} to be used.
	 * 
	 * @param aStoreDescriptor The {@link TrustStoreDescriptor} pointing to your
	 *        {@link KeyStore}.
	 * 
	 * @throws OpenException thrown in case something went wrong.
	 */
	void open( Url aBaseUrl, TrustStoreDescriptor aStoreDescriptor ) throws OpenException;

	/**
	 * {@inheritDoc}
	 */
	@Override
	default HttpRestClient withDisableRequestCorrelation() {
		disableRequestCorrelation();
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default HttpRestClient withDisableSessionCorrelation() {
		disableSessionCorrelation();
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default HttpRestClient withEnableRequestCorrelation() {
		enableRequestCorrelation();
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default HttpRestClient withEnableSessionCorrelation() {
		enableSessionCorrelation();
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default HttpRestClient withRequestCorrelation( boolean hasRequestCorrelation ) {
		setRequestCorrelation( hasRequestCorrelation );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default HttpRestClient withSessionCorrelation( boolean hasSessionCorrelation ) {
		setSessionCorrelation( hasSessionCorrelation );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default HttpRestClient withTrustStoreDescriptor( TrustStoreDescriptor aStoreDescriptor ) {
		setTrustStoreDescriptor( aStoreDescriptor );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default HttpRestClient withUserAgent( String aUserAgent ) {
		setUserAgent( aUserAgent );
		return this;
	}
}
