// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import java.io.InputStream;
import java.net.InetSocketAddress;

import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpServerResponse;
import org.refcodes.net.HttpStatusException;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.Url;

/**
 * Extends a {@link RestServer} to be used as loopback device e.g. for testing
 * purposes such as testing your {@link RestRequestObserver} implementations.
 * Use the
 * {@link #onHttpRequest(InetSocketAddress, InetSocketAddress, HttpMethod, Url, RequestHeaderFields, InputStream, HttpServerResponse)}
 * method to simulate REST requests on the {@link LoopbackRestServer}. An
 * registered {@link RestEndpoint} instances being targeted at will be invoked
 * accordingly.
 */
public interface LoopbackRestServer extends RestServer {

	/**
	 * Invoke this method to simulate a REST request to be handled by the
	 * according registered {@link RestEndpoint} instances:
	 *
	 * @param aLocalAddress the local address
	 * @param aClientAddress the client address
	 * @param aHttpMethod The method for the simulated REST request to be used
	 * @param aUrl The query string part of the request.
	 * @param aRequestHeaderFields The simulated REST request's Header-Fields to
	 *        be used
	 * @param aHttpBody The raw body for the simulated REST request to be used.
	 * @param aHttpServerResponse The {@link HttpServerResponse} to be filled
	 *        with the complete result of processing the request.
	 * @throws HttpStatusException the http status exception
	 */
	public void onHttpRequest( InetSocketAddress aLocalAddress, InetSocketAddress aClientAddress, HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aRequestHeaderFields, String aHttpBody, HttpServerResponse aHttpServerResponse ) throws HttpStatusException;

	/**
	 * Invoke this method to simulate a REST request to be handled by the
	 * according registered {@link RestEndpoint} instances:.
	 *
	 * @param aLocalAddress the local address
	 * @param aClientAddress the client address
	 * @param aHttpMethod The method for the simulated REST request to be used
	 * @param aUrl The query string part of the request.
	 * @param aRequestHeaderFields The simulated REST request's Header-Fields to
	 *        be used
	 * @param aHttpInputStream The HTTP {@link InputStream} representing the
	 *        body for the simulated REST request to be used.
	 * @param aHttpServerResponse The {@link HttpServerResponse} to be filled
	 *        with the complete result of processing the request.
	 * @throws HttpStatusException the http status exception
	 */
	public void onHttpRequest( InetSocketAddress aLocalAddress, InetSocketAddress aClientAddress, HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aRequestHeaderFields, InputStream aHttpInputStream, HttpServerResponse aHttpServerResponse ) throws HttpStatusException;

	/**
	 * {@inheritDoc}
	 */
	@Override
	default LoopbackRestServer withHttpExceptionHandler( HttpExceptionHandler aHttpErrorHandler ) {
		setHttpExceptionHandler( aHttpErrorHandler );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default LoopbackRestServer withHttpExceptionHandling( HttpExceptionHandling aHttpErrorHandling ) {
		setHttpExceptionHandling( aHttpErrorHandling );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default LoopbackRestServer withRealm( String aRealm ) {
		setRealm( aRealm );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default LoopbackRestServer withBaseLocator( String aBaseLocator ) {
		setBaseLocator( aBaseLocator );
		return this;
	}
}
