// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import java.net.MalformedURLException;

import org.refcodes.data.Scheme;
import org.refcodes.net.FormFields;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.Url;

/**
 * Helper class to get the syntactic sugar (from a maintenance point of view)
 * under control. You should actually statically import the
 * {@link HttpRestClientSugar}.
 */
public class RestRequestClientSugar {

	/**
	 * See also {@link HttpRestClient#buildRequest(HttpMethod, String)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aScheme, aHost, aPort, aPath );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, Scheme, String, int, String, FormFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aQueryFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, Scheme, String, int, String, FormFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aQueryFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, Scheme, String, int, String, FormFields, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, Scheme, String, int, String, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, Scheme, String, int, String, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aHeaderFields, aRequest );
	}

	/**
	 * See also {@link HttpRestClient#buildRequest(HttpMethod, String)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aScheme, aHost, aPath );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, Scheme, String, String, FormFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aScheme, aHost, aPath, aQueryFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, Scheme, String, String, FormFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aScheme, aHost, aPath, aQueryFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, Scheme, String, String, FormFields, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aScheme, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, Scheme, String, String, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aScheme, aHost, aPath, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, Scheme, String, String, RequestHeaderFields, Object)}
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aScheme, aHost, aPath, aHeaderFields, aRequest );
	}

	/**
	 * See also {@link HttpRestClient#buildRequest(HttpMethod, String)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aUrl ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aUrl );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, FormFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aLocator, aQueryFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, FormFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, Object aRequest ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aLocator, aQueryFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, FormFields, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * See also {@link HttpRestClient#buildRequest(HttpMethod, String, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aUrl, Object aRequest ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aUrl, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aUrl, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aUrl, aHeaderFields, aRequest );
	}

	/**
	 * See also {@link HttpRestClient#buildRequest(HttpMethod, String)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, String, int, String, FormFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, String, int, String, FormFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, String, int, String, FormFields, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, String, int, String, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, String, int, String, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aHeaderFields, aRequest );
	}

	/**
	 * See also {@link HttpRestClient#buildRequest(HttpMethod, String)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aProtocol, aHost, aPath );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, String, String, FormFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, String, String, FormFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, String, String, FormFields, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, String, String, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aProtocol, aHost, aPath, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, String, String, String, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aProtocol, aHost, aPath, aHeaderFields, aRequest );
	}

	/**
	 * See also {@link HttpRestClient#buildRequest(HttpMethod, String)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Url aUrl ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aUrl );
	}

	/**
	 * See also
	 * {@link HttpRestClient#buildRequest(HttpMethod, Url, RequestHeaderFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().buildRequest( aHttpMethod, aUrl, aHeaderFields );
	}

	/**
	 * See also {@link HttpRestClient#doRequest(HttpMethod, String)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, FormFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aQueryFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, FormFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aQueryFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, FormFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aQueryFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, FormFields, RequestHeaderFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aQueryFields, aHeaderFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, FormFields, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, FormFields, RequestHeaderFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, FormFields,RequestHeaderFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aQueryFields, aHeaderFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, FormFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aQueryFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, RequestHeaderFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aHeaderFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, RequestHeaderFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, RequestHeaderFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aHeaderFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, int, String, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPort, aPath, aResponseObserver );
	}

	/**
	 * See also {@link HttpRestClient#doRequest(HttpMethod, String)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, FormFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aQueryFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, FormFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aQueryFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, FormFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aQueryFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, FormFields, RequestHeaderFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aQueryFields, aHeaderFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, FormFields, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, FormFields, RequestHeaderFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, FormFields,RequestHeaderFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aQueryFields, aHeaderFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, FormFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aQueryFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, RequestHeaderFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aHeaderFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, RequestHeaderFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, RequestHeaderFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aHeaderFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Scheme, String, String, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aScheme, aHost, aPath, aResponseObserver );
	}

	/**
	 * See also {@link HttpRestClient#doRequest(HttpMethod, String)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aUrl ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aUrl );
	}

	/**
	 * See also {@link HttpRestClient#doRequest(HttpMethod, String, FormFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aLocator, aQueryFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, FormFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, Object aRequest ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aLocator, aQueryFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, FormFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aLocator, aQueryFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, FormFields, RequestHeaderFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aLocator, aQueryFields, aHeaderFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, FormFields, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, FormFields, RequestHeaderFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aLocator, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, FormFields,RequestHeaderFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aLocator, aQueryFields, aHeaderFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, FormFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aLocator, aQueryFields, aResponseObserver );
	}

	/**
	 * See also {@link HttpRestClient#doRequest(HttpMethod, String, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aUrl, Object aRequest ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aUrl, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aUrl, Object aRequest, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aUrl, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, RequestHeaderFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aUrl, RequestHeaderFields aHeaderFields ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aUrl, aHeaderFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aUrl, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aUrl, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, RequestHeaderFields, Object, RestResponseObserver)}
	 * *
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aUrl, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aUrl, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, RequestHeaderFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aUrl, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aUrl, aHeaderFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aUrl, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aUrl, aResponseObserver );
	}

	/**
	 * See also {@link HttpRestClient#doRequest(HttpMethod, String)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, FormFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, FormFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, FormFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, FormFields, RequestHeaderFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, FormFields, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, FormFields, RequestHeaderFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, FormFields,RequestHeaderFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, FormFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aQueryFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, RequestHeaderFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aHeaderFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, RequestHeaderFields, Object)}
	 *
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, RequestHeaderFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, RequestHeaderFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aHeaderFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, int, String, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPort, aPath, aResponseObserver );
	}

	/**
	 * See also {@link HttpRestClient#doRequest(HttpMethod, String)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, FormFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, FormFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, FormFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, FormFields, RequestHeaderFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, aHeaderFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, FormFields, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, FormFields, RequestHeaderFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, FormFields,RequestHeaderFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, FormFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aQueryFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, RequestHeaderFields)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aHeaderFields );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, RequestHeaderFields, Object)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aHeaderFields, aRequest );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, RequestHeaderFields, Object, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#DELETE} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, RequestHeaderFields, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aHeaderFields, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, String, String, String, RestResponseObserver)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aProtocol, aHost, aPath, aResponseObserver );
	}

	/**
	 * See also {@link HttpRestClient#doRequest(HttpMethod, Url)}
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestResponse doRequest( HttpMethod aHttpMethod, Url aUrl ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aUrl );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Url, Object, RestResponseObserver)}
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Url aUrl, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aUrl, (RequestHeaderFields) null, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Url, RequestHeaderFields, Object, RestResponseObserver)}
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aUrl, (RequestHeaderFields) aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * See also
	 * {@link HttpRestClient#doRequest(HttpMethod, Url, RequestHeaderFields, RestResponseObserver)}
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	public static RestCallerBuilder doRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return HttpRestClientSingleton.getInstance().doRequest( aHttpMethod, aUrl, (RequestHeaderFields) aHeaderFields, null, aResponseObserver );
	}

	/**
	 * See also {@link HttpRestClient#doRequest(RestRequest)}
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	public static RestResponse doRequest( RestRequest aRequest ) throws HttpResponseException, MalformedURLException {
		return HttpRestClientSingleton.getInstance().doRequest( aRequest );
	}
}
