// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatusAccessor;
import org.refcodes.component.LinkComponent.LinkComponentBuilder;
import org.refcodes.component.OpenException;
import org.refcodes.net.BasicAuthObserver;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpMethodAccessor.HttpMethodBuilder;
import org.refcodes.net.HttpMethodAccessor.HttpMethodProperty;

/**
 * A {@link BasicAuthEndpointBuilder} extends an {@link BasicAuthEndpoint} with
 * builder functionality and adds <code>lambda</code> support for handling the
 * requests addressed to this {@link BasicAuthEndpoint}. The <code>lambda</code>
 * defined as {@link BasicAuthObserver} acts as the single listener to this
 * {@link BasicAuthEndpoint} responsible for handling the requests for which
 * this {@link BasicAuthEndpoint} is responsible.
 * 
 * The locator for which an {@link BasicAuthEndpointBuilder} is responsible for
 * is defined by the {@link BasicAuthEndpointBuilder}'s Locator-Pattern:
 * 
 * A single asterisk ("*") matches zero or more characters within a locator
 * name. A double asterisk ("**") matches zero or more characters across
 * directory levels. A question mark ("?") matches exactly one character within
 * a locator name.
 * 
 * The single asterisk ("*"), the double asterisk ("**") and the question mark
 * ("?") we refer to as wildcard: You get an array with the substitutes of the
 * wildcards using the method {@link RestRequestEvent#getWildcardReplacements()}
 * .
 * 
 * You may name a wildcard by prefixing it with "${someWildcardName}". For
 * example a named wildcard may look as follows: "${arg1}=*" or "${arg2}=**" or
 * "${arg3}=?". You can get the text substituting a named wildcard using the
 * method {@link RestRequestEvent#getWildcardReplacement(String)}.
 * 
 * For ease of use, a named wildcard with single asterisk ("*") such as
 * "${arg1}=*" can be abbreviated as "${arg1}".
 */
public interface BasicAuthEndpointBuilder extends BasicAuthEndpoint, HttpMethodProperty, HttpMethodBuilder<BasicAuthEndpointBuilder>, LinkComponentBuilder<BasicAuthEndpointBuilder>, ConnectionStatusAccessor {

	/**
	 * {@inheritDoc}
	 */
	@Override
	default BasicAuthEndpointBuilder withHttpMethod( HttpMethod aHttpMethod ) {
		setHttpMethod( aHttpMethod );
		return this;
	}

	/**
	 * Sets the resource locator as of the Builder-Pattern. The locator may
	 * contain wildcards as known from file-systems as follows:
	 * 
	 * A single asterisk ("*") matches zero or more characters within a locator
	 * name. A double asterisk ("**") matches zero or more characters across
	 * directory levels. A question mark ("?") matches exactly one character
	 * within a locator name.
	 * 
	 * You may name a wildcard by prefixing it with "${someWildcardName}". For
	 * example a named wildcard may look as follows: "${arg1}=*" or "${arg2}=**"
	 * or "${arg3}=?". You can get the text substituting a named wildcard using
	 * the method {@link RestRequestEvent#getWildcardReplacement(String)}.
	 * 
	 * For ease of use, a named wildcard with single asterisk ("*") such as
	 * "${arg1}=*" can be abbreviated as "${arg1}".
	 *
	 * @param aLocatorPattern the locator pattern
	 * @return The {@link BasicAuthEndpoint} builder to continue configuration
	 *         (as of the Builder-Pattern).
	 */
	default BasicAuthEndpointBuilder withLocatorPattern( String aLocatorPattern ) {
		setLocatorPattern( aLocatorPattern );
		return this;
	}

	/**
	 * Sets the resource locator. The locator may contain wildcards as known
	 * from file-systems as follows:
	 * 
	 * A single asterisk ("*") matches zero or more characters within a locator
	 * name. A double asterisk ("**") matches zero or more characters across
	 * directory levels. A question mark ("?") matches exactly one character
	 * within a locator name.
	 * 
	 * You may name a wildcard by prefixing it with "${someWildcardName}". For
	 * example a named wildcard may look as follows: "${arg1}=*" or "${arg2}=**"
	 * or "${arg3}=?". You can get the text substituting a named wildcard using
	 * the method {@link RestRequestEvent#getWildcardReplacement(String)}.
	 * 
	 * For ease of use, a named wildcard with single asterisk ("*") such as
	 * "${arg1}=*" can be abbreviated as "${arg1}".
	 *
	 * @param aLocatorPattern the new locator pattern
	 */
	void setLocatorPattern( String aLocatorPattern );

	/**
	 * Retrieves the {@link BasicAuthObserver} to which any requests are
	 * delegated upon invocation of the
	 * {@link #onBasicAuthRequest(java.net.InetSocketAddress, java.net.InetSocketAddress, HttpMethod, String, org.refcodes.net.BasicAuthCredentials, String)}
	 * method.
	 * 
	 * @return The (user defined) {@link BasicAuthObserver} to handle requests.
	 */
	BasicAuthObserver getRequestObserver();

	/**
	 * Sets the {@link BasicAuthObserver} to which any requests are delegated
	 * upon invocation of the
	 * {@link #onBasicAuthRequest(java.net.InetSocketAddress, java.net.InetSocketAddress, HttpMethod, String, org.refcodes.net.BasicAuthCredentials, String)}
	 * method.
	 * 
	 * @param aLambda The (user defined) {@link BasicAuthObserver} to handle
	 *        requests, feel free to code it as <code>lambda</code> expression!
	 */
	void setRequestObserver( BasicAuthObserver aLambda );

	/**
	 * Builder method for setting the {@link BasicAuthObserver}.
	 * 
	 * @param aLambda The (user defined) {@link BasicAuthObserver} to handle
	 *        requests, feel free to code it as <code>lambda</code> expression
	 * 
	 * @return The {@link BasicAuthEndpointBuilder} for the sake of a fluent
	 *         API.
	 */
	default BasicAuthEndpointBuilder withRequestObserver( BasicAuthObserver aLambda ) {
		setRequestObserver( aLambda );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default BasicAuthEndpointBuilder withOpen() throws OpenException {
		open();
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default BasicAuthEndpointBuilder withClose() throws CloseException {
		close();
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default BasicAuthEndpointBuilder withCloseQuietly() {
		closeQuietly();
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default BasicAuthEndpointBuilder withCloseIn( int aCloseInMillis ) {
		closeIn( aCloseInMillis );
		return this;
	}

}
