// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import java.net.MalformedURLException;

import org.refcodes.data.Scheme;
import org.refcodes.net.FormFields;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.Url;

/**
 * Helper interface to keep the huge amount of convenience methods under
 * control.
 */
public interface RestPostClient extends RestRequestClient {

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, int aPort, String aPath ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aQueryFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aQueryFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aQueryFields, aHeaderFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, int aPort, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aHeaderFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, String aPath ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPath );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPath, aQueryFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPath, aQueryFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPath, aQueryFields, aHeaderFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPath, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPath, aHeaderFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aScheme, aHost, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aUrl The targeted URL locating the addressed resource..
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildPost( String aUrl ) throws MalformedURLException {
		return buildRequest( HttpMethod.POST, aUrl );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 * 
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildPost( String aLocator, FormFields aQueryFields ) throws MalformedURLException {
		return buildRequest( HttpMethod.POST, aLocator, aQueryFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildPost( String aLocator, FormFields aQueryFields, Object aRequest ) throws MalformedURLException {
		return buildRequest( HttpMethod.POST, aLocator, aQueryFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment..
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildPost( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws MalformedURLException {
		return buildRequest( HttpMethod.POST, aLocator, aQueryFields, aHeaderFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment..
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildPost( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws MalformedURLException {
		return buildRequest( HttpMethod.POST, aLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aUrl The targeted URL locating the addressed resource..
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildPost( String aUrl, Object aRequest ) throws MalformedURLException {
		return buildRequest( HttpMethod.POST, aUrl, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aUrl The targeted URL locating the addressed resource..
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildPost( String aUrl, RequestHeaderFields aHeaderFields ) throws MalformedURLException {
		return buildRequest( HttpMethod.POST, aUrl, aHeaderFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aUrl The targeted URL locating the addressed resource..
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildPost( String aUrl, RequestHeaderFields aHeaderFields, Object aRequest ) throws MalformedURLException {
		return buildRequest( HttpMethod.POST, aUrl, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) including the specific
	 *        part (such as "//" for "http://") to be used for the destination
	 *        URL: For HTTP, provide "http://", for HTTPS, provide "https://".
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aHeaderFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aHeaderFields );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( HttpMethod.POST, aProtocol, aHost, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Url aUrl ) {
		return buildRequest( HttpMethod.POST, aUrl );
	}

	/**
	 * Prepares a POST request builder with the common attributes and returns
	 * the according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildPost( Url aUrl, RequestHeaderFields aHeaderFields ) {
		return buildRequest( HttpMethod.POST, aUrl, aHeaderFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestResponse} as of your request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aQueryFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aQueryFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aQueryFields, aRequest, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aQueryFields, aHeaderFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aQueryFields, aHeaderFields, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aQueryFields, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aRequest, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aHeaderFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aHeaderFields, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPort, aPath, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aQueryFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aQueryFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aQueryFields, aRequest, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aQueryFields, aHeaderFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aQueryFields, aHeaderFields, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aQueryFields, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aRequest, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aHeaderFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aHeaderFields, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Scheme aScheme, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aScheme, aHost, aPath, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aUrl The targeted URL locating the addressed resource..
	 * 
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doPost( String aUrl ) throws HttpResponseException, MalformedURLException {
		return doRequest( HttpMethod.POST, aUrl );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment..
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doPost( String aLocator, FormFields aQueryFields ) throws HttpResponseException, MalformedURLException {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment..
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doPost( String aLocator, FormFields aQueryFields, Object aRequest ) throws HttpResponseException, MalformedURLException {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment..
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doPost( String aLocator, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, aRequest, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment..
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doPost( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException, MalformedURLException {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, aHeaderFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment..
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doPost( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException, MalformedURLException {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment..
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doPost( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment..
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doPost( String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, aHeaderFields, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment..
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doPost( String aLocator, FormFields aQueryFields, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return doRequest( HttpMethod.POST, aLocator, aQueryFields, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aUrl The targeted URL locating the addressed resource..
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doPost( String aUrl, Object aRequest ) throws HttpResponseException, MalformedURLException {
		return doRequest( HttpMethod.POST, aUrl, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aUrl The targeted URL locating the addressed resource..
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doPost( String aUrl, Object aRequest, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return doRequest( HttpMethod.POST, aUrl, aRequest, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aUrl The targeted URL locating the addressed resource..
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doPost( String aUrl, RequestHeaderFields aHeaderFields ) throws HttpResponseException, MalformedURLException {
		return doRequest( HttpMethod.POST, aUrl, aHeaderFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aUrl The targeted URL locating the addressed resource..
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doPost( String aUrl, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException, MalformedURLException {
		return doRequest( HttpMethod.POST, aUrl, aHeaderFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aUrl The targeted URL locating the addressed resource..
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doPost( String aUrl, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return doRequest( HttpMethod.POST, aUrl, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aUrl The targeted URL locating the addressed resource..
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doPost( String aUrl, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return doRequest( HttpMethod.POST, aUrl, aHeaderFields, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aUrl The targeted URL locating the addressed resource..
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doPost( String aUrl, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return doRequest( HttpMethod.POST, aUrl, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aRequest, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aHeaderFields, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aQueryFields, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aRequest, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aHeaderFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aHeaderFields, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPort, aPath, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aRequest, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aHeaderFields, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aQueryFields, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aRequest, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aHeaderFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aHeaderFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aHeaderFields, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( String aProtocol, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aProtocol, aHost, aPath, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Url aUrl ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aUrl );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Url aUrl, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aUrl, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Url aUrl, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aUrl, aRequest, aResponseObserver );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Url aUrl, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aUrl, aHeaderFields );
	}

	/**
	 * Sends a POST request with the common attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doPost( Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( HttpMethod.POST, aUrl, aHeaderFields, aRequest );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aUrl, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a POST request with the common attributes and returns the
	 * according {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCaller}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by your request.
	 */
	default RestCallerBuilder doPost( Url aUrl, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( HttpMethod.POST, aUrl, aHeaderFields, aResponseObserver );
	}
}
