// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

/**
 * 
 * <h1>What is this package for?</h1>
 * 
 * With this artifact you easily create you RESTful servers and REST clients.
 * The <code>refcodes-rest</code> artifact lets you do it the
 * <code>Bare-Metal</code> way, or the <code>syntactic sugar</code> (see
 * <code>client sugar</code> or <code>server sugar</code>) way (meaning the use
 * of statically imported methods):
 * 
 * <h2>RESTful server</h2>
 * 
 * <ol>
 * <li>Instantiate the RESTful server
 * <li>Register your lambda expressions
 * <li>Start the RESTful server
 * </ol>
 * 
 * Within three simple steps you implement your lightweight <code>Java</code>
 * based <code>RESTful</code> server. Below, you see the three steps with the
 * help of a little <code>syntactic sugar</code>:
 * 
 * <pre>
 * {@code
 * // STEP 1: We use a singleton with syntactic sugar instead of instantiating a HttpRestServer:
 * import static org.refcodes.rest.HttpRestServerSugar.*;
 * 	... 
 * 	public static void main( String[] args ) {
 * 		// STEP 2: Using syntactic sugar, we register our lambda expression:
 * 		onGet( "/say/${name}=*", ( aResponse ) -> {
 * 			String name = aResponse.getWildcardReplacement( "name" );
 * 			aResponse.getHeaderFields().withContentType( MediaType.APPLICATION_JSON ).withAddCookie( "greeting", "Hello " + name + "!" );	
 * 			return "Hello " + name + "!" ;
 * 		} ).open();
 * 		// STEP 3: We open the HttpRestServer singleton on port 8080 using our syntactic sugar:
 * 		open( 8080 );
 * 	}
 * 	...
 * }
 * </pre>
 * 
 * The <a href=
 * "https://bitbucket.org/refcodes/funcodes-tinyrestful/raw/master/src/main/java/club/funcodes/tinyrestful/TinyRestfulServer.java">
 * TinyRestfulServer</a> demo application uses <code>syntactic sugar</code> for
 * setting up a <code>RESTful</code> server including command line arguments
 * parsing.
 * 
 * <h2>REST client</h2>
 * 
 * <ol>
 * <li>Instantiate the REST client
 * <li>Register your lambda expressions
 * <li>Fire the client's REST request
 * </ol>
 * 
 * Again, within three simple steps you implement your lightweight
 * <code>Java</code> based <code>REST</code> client. Below you see the three
 * steps with the help of a little <code>syntactic sugar</code>:
 * 
 * <pre>
 * {@code
 * // STEP 1: We use a singleton with syntactic sugar instead of instantiating a HttpRestClient: 
 * import static org.refcodes.rest.HttpRestClientSugar.*;
 * 	... 
 * 	public static void main( String[] args ) {
 * 		// STEP 2: Using syntactic sugar, we define our caller, including the response listener:
 * 		doGet( "http://mydomain:8080/say", ( aResponse ) -> {
 * 			... = aResponse.getResponse( SomeType.class );
 * 		} ).withRequest( ... ).open();
 * 		// STEP 3: We opened the caller so it fires the request to port 8080 of domain "mydomain"
 * 	}
 * 	...
 * }
 * </pre>
 * 
 * <h1>How do I get set up?</h1>
 * 
 * To get up and running, include the following dependency (without the three
 * dots "...") in your <code>pom.xml</code>:
 * 
 * <pre>
 * {@code
 * <dependencies>
 * 	...
 * 	<dependency>
 * 		<artifactId>refcodes-rest</artifactId>
 * 		<groupId>org.refcodes</groupId>
 * 		<version>x.y.z</version>
 * 	</dependency>
 * 	...
 * </dependencies>
 * }
 * </pre>
 * 
 * (please refer to Maven Central at
 * "http://search.maven.org/#search|ga|1|g%3A%22org.refcodes%22" for the most
 * current version)
 * <p>
 * If you want the framework to interoperate with <code>SLF4J</code>
 * (http://www.slf4j.org/) logging, you may instead add the following
 * dependencies to your <code>pom.xml</code>:
 * 
 * <pre>
 * {@code
 * <dependencies>
 * 	...
 * 	<dependency>
 * 		<artifactId>refcodes-rest</artifactId>
 * 		<groupId>org.refcodes</groupId>
 * 		<version>x.y.z</version>
 * 	</dependency>
 * 	<dependency>
 * 		<groupId>org.refcodes</groupId>
 * 		<artifactId>refcodes-logger-alt-slf4j</artifactId>
 * 		<version>x.y.z</version>
 * 	</dependency>
 * 	...
 * </dependencies>
 * }
 * </pre>
 * 
 * (please refer to Maven Central at
 * "http://search.maven.org/#search|ga|1|g%3A%22org.refcodes%22" for the most
 * current version)
 * <p>
 * The artifact is hosted directly at Maven Central (http://search.maven.org).
 * Jump straight to the source codes at Bitbucket
 * (https://bitbucket.org/refcodes/refcodes-rest). Read the artifact's javadoc
 * at javadoc.io (http://www.javadoc.io/doc/org.refcodes/refcodes-rest).
 * 
 * <h2>How do I get started with the RESTful server?</h2>
 * 
 * Above you saw an example on how to setup your own <code>RESTful</code>
 * service using <code>syntactic sugar</code>. One drawback of using
 * <code>syntactic sugar</code> is that we can only make use of the one
 * {@link org.refcodes.rest.HttpRestServerSingleton} (as of this
 * <code>syntactic sugar</code> being the statically imported methods),
 * preventing us from running multiple {@link org.refcodes.rest.HttpRestServer}
 * instances on different ports in one <code>Java</code> application.
 * <p>
 * Lets do it the <code>Bare-Metal</code> way, which is not very complicated
 * either, and which lets us instantiate as many
 * {@link org.refcodes.rest.HttpRestServer} instances as we want:
 * 
 * <pre>
 * {@code
 * ...
 * public static void main( String[] args ) {
 * 	// STEP 1: We instantiate our HttpRestServer:
 * 	HttpRestServer theRestServer = new HttpRestServerImpl();
 * 	// STEP 2: We register our lambda expression:
 * 	theRestServer.onGet( "/say/${name}=*", ( aResponse ) -> {
 * 		String name = aResponse.getWildcardReplacement( "name" );
 * 		aResponse.getHeaderFields().withContentType( MediaType.APPLICATION_JSON ).withAddCookie( "greeting", "Hello " + name + "!" );
 * 		return "Hello " + name + "!" ;
 * 	} ).open();
 * 	// STEP 3: We open the HttpRestServer instance on port 8080:
 * 	theRestServer.open( 8080 );
 * }
 * ...
 * }
 * </pre>
 * 
 * <h3>The Locator-Pattern</h3>
 * 
 * Did you notice the Locator-Pattern <code>"/say/${name}=*"</code> above when
 * registering your <code>lambda</code>? Subscribing your <code>lambda</code>
 * expressions for incoming <code>REST</code> requests on specific
 * <code>locators</code>, you may use a common <code>wildcard</code> syntax to
 * define the <code>lambda</code>'s <code>locator pattern</code>:
 * 
 * <ul>
 * <li>A single asterisk ("*") matches zero or more characters within a locator
 * name.
 * <li>A double asterisk ("**") matches zero or more characters across directory
 * levels.
 * <li>A question mark ("?") matches exactly one character within a locator
 * name.
 * </ul>
 * 
 * The single asterisk (<code>*</code>), the double asterisk (<code>**</code>)
 * and the question mark (<code>?</code>) we refer to as <code>wildcard</code>:
 * You get an array with all the substitutes of the <code>wildcards</code> using
 * the method <code>RestRequestEvent#getWildcardReplacements()</code>.
 * <p>
 * You may name a <code>wildcard</code> by prefixing it with
 * "<code>${someWildcardName}=</code>". For example a named
 * <code>wildcard</code> may look as follows: "<code>${arg1}=*</code>" or
 * "<code>${arg2}=**</code>" or "<code>${arg3}=?</code>" or as of the example
 * above <code>"/say/${name}=*"</code>. When your "<code>lambda</code>" is being
 * invoked, you can retrieve the <code>wildcard</code> substitution by the name
 * of the <code>wildcard</code> which has been substituted (by parts of the
 * incoming locator). You can get the text substituting a named
 * <code>wildcard</code> using the method
 * <code>RestRequestEvent#getWildcardReplacement(String)</code>.
 * 
 * <h3>The RESTful server's bits and pieces</h3>
 * 
 * <ul>
 * <li>{@link org.refcodes.rest.RestServer}: It acts as the target for clients
 * issuing REST requests. {@link org.refcodes.rest.RestEndpointBuilder})
 * instances, most easily being created with the
 * RestServer#subscribeObserver(HttpMethod, String, RestRequestObserver) or the
 * like methods, are registered as listeners to the
 * {@link org.refcodes.rest.RestServer}. The
 * {@link org.refcodes.rest.RestServer} fires
 * {@link org.refcodes.rest.RestRequestEvent} events to the
 * {@link org.refcodes.rest.RestRequestObserver}s of a
 * {@link org.refcodes.rest.RestEndpoint} dedicated to an according
 * locator(pattern) for a specific {@link org.refcodes.net.HttpMethod}.
 * <li>{@link org.refcodes.rest.HttpRestServer}: It extends a
 * {@link org.refcodes.rest.RestServer} to be capable of opening a server socket
 * on the local host with the provided port number via #open(Integer) or with an
 * additional maximum number of connections via #open(Integer, int) . A
 * {@link org.refcodes.rest.HttpRestServer} can be shutdown via #close().
 * <li>{@link org.refcodes.rest.HttpRestServerImpl}: Implementation of the
 * {@link org.refcodes.rest.HttpRestServer} interface using the HttpServer
 * defined in the com.sun.net.httpserver artifact. The
 * {@link org.refcodes.rest.HttpRestServer} can also be implemented with other
 * HTTP servers under the hood, use the
 * {@link org.refcodes.rest.AbstractRestServer} as super class of your own
 * implementation to easily do so.
 * <li>{@link org.refcodes.rest.RestEndpoint}: A
 * {@link org.refcodes.rest.RestEndpoint} subscribes to a
 * {@link org.refcodes.rest.RestServer}) and defines the target for a REST
 * request. Therefore the {@link org.refcodes.rest.RestEndpoint} describes the
 * {@link org.refcodes.net.HttpMethod}, the locator (pattern) to which to
 * respond as well as a {@link org.refcodes.rest.RestRequestObserver}
 * responsible for processing the request. The
 * {@link org.refcodes.rest.RestRequestObserver} is invoked as soon as a request
 * with the given {@link org.refcodes.net.HttpMethod} for a locator matching the
 * given Locator-Pattern is being processed by the
 * {@link org.refcodes.rest.RestServer}).
 * <li>{@link org.refcodes.rest.RestEndpointBuilder}): A
 * {@link org.refcodes.rest.RestEndpointBuilder}) extends a
 * {@link org.refcodes.rest.RestEndpoint} with builder functionality and adds
 * lambda support for handling the request addressed to this
 * {@link org.refcodes.rest.RestEndpoint}. The lambda defined as
 * {@link org.refcodes.rest.RestRequestObserver} acts as the single listener to
 * this {@link org.refcodes.rest.RestEndpoint} responsible for handling the
 * request for which this {@link org.refcodes.rest.RestEndpoint} is responsible.
 * <li>{@link org.refcodes.rest.RestRequestObserver}: It can be coded using the
 * lambda syntax and processes a request on a given locator for a given
 * {@link org.refcodes.net.HttpMethod}. The
 * {@link org.refcodes.rest.RestRequestObserver} is working with a context (the
 * {@link org.refcodes.rest.RestRequestEvent}).
 * <li>{@link org.refcodes.rest.HttpRestServerSugar}: The syntactic sugar for
 * setting up your RESTful service as quickly as possible ("import static
 * org.refcodes.rest.HttpRestServerSugar.*;").
 * </ul>
 * 
 * <h2>How do I get started with the REST client?</h2>
 * 
 * Above you saw an example on how to setup your own <code>REST</code> client
 * using <code>syntactic sugar</code>. One drawback of using
 * <code>syntactic sugar</code> is that we can only make use of the one
 * {@link org.refcodes.rest.HttpRestClientSingleton} (as of this
 * <code>syntactic sugar</code> being the statically imported methods),
 * preventing us from running multiple {@link org.refcodes.rest.HttpRestClient}
 * instances in one <code>Java</code> application (which is actually no real
 * drawback, as the {@link org.refcodes.rest.HttpRestClientSingleton} can fire
 * at any HTTP or HTTPS targets you wish to connect to).
 * <p>
 * Lets do it the <code>Bare-Metal</code> way, which is not very complicated
 * either, and which lets us instantiate as many
 * {@link org.refcodes.rest.HttpRestClient} instances as we want:
 * 
 * <pre>
 * {@code
 * ...
 * public static void main( String[] args ) {
 * 	// STEP 1: We instantiate our HttpRestClient:
 * 	HttpRestClient theRestClient = new HttpRestClientImpl();
 * 	// STEP 2: We register our lambda expression:
 * 	theRestClient.doRequest( HttpMethod.POST, "http://mydomain:8080/say", ( aResponse ) -> {
 * 		String theResponse = aResponse.getResponse( String.class );
 * 	} ).withRequest( ... ).open();
 * 	// STEP 3: We opened the caller so it fires the request to port 8080 of domain "mydomain"
 * }
 * ...
 * }
 * </pre>
 * 
 * <h3>The REST client's bits and pieces</h3>
 * 
 * <ul>
 * <li>{@link org.refcodes.rest.RestClient}: It acts as the origin for clients
 * issuing REST requests. {@link org.refcodes.rest.RestCallerBuilder} instances,
 * most easily being created with the RestClient#doRequest( HttpMethod ,
 * aLocator, aResponseObserver ) or the like methods, are registered as
 * listeners to the {@link org.refcodes.rest.RestClient}'s request, waiting for
 * the response. The {@link org.refcodes.rest.RestClient} fires
 * {@link org.refcodes.rest.RestResponseEvent} events to the
 * {@link org.refcodes.rest.RestResponseObserver} of the
 * {@link org.refcodes.rest.RestCaller} dedicated to the according request.
 * <li>{@link org.refcodes.rest.HttpRestClient}: It extends a
 * {@link org.refcodes.rest.RestClient} to be capable of doing HTTP (HTTPS). A
 * {@link org.refcodes.rest.HttpRestClient} can be shutdown via #close().
 * <li>{@link org.refcodes.rest.HttpRestClientImpl}: Implementation of the
 * {@link org.refcodes.rest.HttpRestClient} interface using the
 * HttpURLConnection
 * (https://docs.oracle.com/javase/8/docs/api/java/net/HttpURLConnection.html)
 * defined in the java.net package. The {@link org.refcodes.rest.HttpRestClient}
 * can also be implemented with other HTTP connectors under the hood, use the
 * {@link org.refcodes.rest.AbstractRestClient} as super class of your own
 * implementation to easily do so.
 * <li>{@link org.refcodes.rest.RestCaller}: A
 * {@link org.refcodes.rest.RestCaller} subscribes to a
 * {@link org.refcodes.rest.RestClient} (
 * {@link org.refcodes.rest.HttpRestClient})'s request and defines the target
 * for a REST request's response. Therefore the
 * {@link org.refcodes.rest.RestCaller} describes the
 * {@link org.refcodes.rest.RestResponseObserver} responsible for processing the
 * request's response. The {@link org.refcodes.rest.RestResponseObserver} is
 * invoked as soon as a response for the according request is being received by
 * the {@link org.refcodes.rest.RestClient} (
 * {@link org.refcodes.rest.HttpRestClient}).
 * <li>{@link org.refcodes.rest.RestCallerBuilder}: A
 * {@link org.refcodes.rest.RestCallerBuilder} extends a
 * {@link org.refcodes.rest.RestCaller} with builder functionality and adds
 * lambda support for handling the responses addressed to this
 * {@link org.refcodes.rest.RestCaller}. The lambda defined as RequestObserver
 * acts as the single listener to this {@link org.refcodes.rest.RestCaller}
 * responsible for handling the response for which this
 * {@link org.refcodes.rest.RestCaller} is responsible.
 * <li>{@link org.refcodes.rest.RestResponseObserver}: It can be coded using the
 * lambda syntax and processes a request's response. The
 * {@link org.refcodes.rest.RestResponseObserver} is working with a context (the
 * {@link org.refcodes.rest.RestResponseEvent}).
 * <li>{@link org.refcodes.rest.HttpRestClientSugar}: The syntactic sugar for
 * setting up your REST client as quickly as possible ("import static
 * org.refcodes.rest.HttpRestClientSugar.*;").
 * </ul>
 *
 * <h2>Examples</h2>
 * 
 * Please refer to the example code at
 * "https://bitbucket.org/refcodes/refcodes-rest/src/master/src/test/java/org/refcodes/rest".
 * 
 * Also see
 * "https://bitbucket.org/refcodes/refcodes-rest/src/master/src/test/java/org/refcodes/rest"
 */
package org.refcodes.rest;