// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetSocketAddress;
import java.net.URI;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.Base64;
import java.util.concurrent.ExecutorService;

import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLParameters;
import javax.net.ssl.TrustManagerFactory;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatus;
import org.refcodes.component.OpenException;
import org.refcodes.controlflow.ControlFlowUtility;
import org.refcodes.controlflow.ThreadingModel;
import org.refcodes.data.Delimiter;
import org.refcodes.data.LatencySleepTime;
import org.refcodes.data.Literal;
import org.refcodes.data.Scheme;
import org.refcodes.exception.BugException;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.exception.MarshalException;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.RuntimeLoggerFactorySingleton;
import org.refcodes.net.FormMediaTypeFactory;
import org.refcodes.net.JsonMediaTypeFactory;
import org.refcodes.net.XmlMediaTypeFactory;
import org.refcodes.net.AuthType;
import org.refcodes.net.BadResponseException;
import org.refcodes.net.BasicAuthCredentials;
import org.refcodes.net.BasicAuthCredentialsImpl;
import org.refcodes.net.BasicAuthObserver;
import org.refcodes.net.BasicAuthRequiredException;
import org.refcodes.net.BasicAuthResponse;
import org.refcodes.net.ContentType;
import org.refcodes.net.HeaderField;
import org.refcodes.net.HeaderFields;
import org.refcodes.net.HttpBodyMap;
import org.refcodes.net.HttpBodyMapImpl;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpServerResponse;
import org.refcodes.net.HttpServerResponseImpl;
import org.refcodes.net.HttpStatusCode;
import org.refcodes.net.HttpStatusException;
import org.refcodes.net.HttpsConnectionRequestObserver;
import org.refcodes.net.MediaType;
import org.refcodes.net.MediaTypeFactory;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.RequestHeaderFieldsImpl;
import org.refcodes.net.TextMediaTypeFactory;
import org.refcodes.net.TransportLayerProtocol;
import org.refcodes.net.UnsupportedMediaTypeException;
import org.refcodes.net.Url;
import org.refcodes.net.Url.UrlBuilder;
import org.refcodes.net.UrlBuilderImpl;
import org.refcodes.security.KeyStoreDescriptor;

import com.sun.net.httpserver.Authenticator;
import com.sun.net.httpserver.Authenticator.Result;
import com.sun.net.httpserver.Authenticator.Retry;
import com.sun.net.httpserver.Headers;
import com.sun.net.httpserver.HttpContext;
import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpPrincipal;
import com.sun.net.httpserver.HttpServer;
import com.sun.net.httpserver.HttpsConfigurator;
import com.sun.net.httpserver.HttpsParameters;
import com.sun.net.httpserver.HttpsServer;

/**
 * Implementation of the {@link HttpRestServer} interface using the
 * {@link HttpServer} defined in the <code>com.sun.net.httpserver</code>
 * package.
 * 
 * The {@link HttpRestServerImpl} is being initialized with some common
 * {@link MediaTypeFactory} instances (as implemented by the
 * {@link AbstractRestServer}). At the time of writing this document the
 * {@link MediaTypeFactory} instances being pre-configured are:
 * 
 * <ul>
 * <li>{@link JsonMediaTypeFactory}</li>
 * <li>{@link XmlMediaTypeFactory}</li>
 * <li>{@link TextMediaTypeFactory}</li>
 * <li>{@link FormMediaTypeFactory}</li>
 * </ul>
 * 
 * The {@link HttpRestServerImpl} supports HTTP as well as HTTPS protocols as
 * being based on the {@link HttpServer} as well as on the {@link HttpsServer}.
 * 
 * For opening up an HTTPS connection, refer to the methods such as
 * {@link #open(String, KeyStoreDescriptor, int)} or
 * {@link #open(KeyStoreDescriptor, int)} and the like.
 */
public class HttpRestServerImpl extends AbstractRestServer implements HttpRestServer {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	protected static final String CONTEXT_PATH = Delimiter.PATH.getChar() + "";
	private static final String ANONYMOUS = "anonymous";
	private static final int NO_RESPONSE_BODY = -1;
	private static final int CHUNCKED_ENCODING = 0;

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private HttpServer _httpServer = null;
	private int _port = -1;
	private HttpsConnectionRequestObserver _httpsConnectionRequestObserver = null;
	private ExecutorService _executorService;
	private ConnectionStatus _connectionStatus = ConnectionStatus.NONE;
	private HttpBasicAuthenticator _httpBasicAuthenticator = null;
	private HttpContext _httpContext;
	private Scheme _scheme = null;
	private String _protocol = null;
	private KeyStoreDescriptor _keyStoreDescriptor = null;
	private int _maxConnections = -1;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link HttpRestServerImpl}. Use {@link #open(int)} or
	 * similar to make it listen on your port.
	 * 
	 * The provided {@link ThreadingModel} defines whether to use a
	 * single-threaded or a multi-threaded threading model. Threads are created
	 * as daemon threads. For more control on thread generation use the
	 * constructor {@link #HttpRestServerImpl(ExecutorService)}.
	 * 
	 * @param aThreadingModel Specifies to either use a single-threaded
	 *        threading model (no {@link ExecutorService}) or a multi-threaded
	 *        threading model with a default {@link ExecutorService} as of
	 *        {@link ControlFlowUtility#createCachedExecutorService(boolean)}.
	 */
	public HttpRestServerImpl( ThreadingModel aThreadingModel ) {
		this( aThreadingModel == ThreadingModel.SINGLE ? null : ControlFlowUtility.createCachedExecutorService( true ) );
	}

	/**
	 * Constructs a {@link HttpRestServerImpl}. Use {@link #open(int)} or
	 * similar to make it listen on your port.
	 * 
	 * Uses a multi threaded threading model with a default
	 * {@link ExecutorService} as of
	 * {@link ControlFlowUtility#createCachedExecutorService(boolean)}. Threads
	 * are created as daemon threads. For more control on thread generation use
	 * the constructor {@link #HttpRestServerImpl(ExecutorService)}.
	 */
	public HttpRestServerImpl() {
		this( ControlFlowUtility.createCachedExecutorService( true ) );
	}

	/**
	 * Constructs a {@link HttpRestServerImpl}. Use {@link #open(int)} or
	 * similar to make it listen on your port.
	 * 
	 * Uses a multi threaded threading model.
	 *
	 * @param aExecutorService An executor service to be used when creating
	 *        {@link Thread}s.
	 */
	public HttpRestServerImpl( ExecutorService aExecutorService ) {
		super( aExecutorService );
		_executorService = aExecutorService;
		_httpExceptionHandler = new DefaultErrorHandler();
	}

	// /////////////////////////////////////////////////////////////////////////
	// LIFECYCLE:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void close() throws CloseException {
		if ( _connectionStatus != ConnectionStatus.OPENED ) {
			throw new CloseException( "Connection is in status <" + _connectionStatus + ">. Open the connection before closing!" );
		}
		try {
			if ( _httpServer != null ) {
				_httpServer.stop( LatencySleepTime.MIN.getMilliseconds() / 1000 );
				_httpServer.removeContext( CONTEXT_PATH );
				_httpServer = null;
			}
		}
		finally {
			_connectionStatus = ConnectionStatus.CLOSED;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void open( String aProtocol, KeyStoreDescriptor aStoreDescriptor, int aPort, int aMaxConnections ) throws OpenException {

		if ( aPort < 0 ) aPort = getPort();
		if ( aPort < 0 ) {
			throw new OpenException( "You must provide a valid port via 'setPort( aPort )' before you can invoke a port-less 'open' method!" );
		}
		_port = aPort;
		if ( aProtocol == null ) aProtocol = toProtocol();
		Scheme theScheme = Scheme.fromProtocol( aProtocol );
		if ( theScheme == Scheme.HTTPS ) {
			aProtocol = TransportLayerProtocol.TLS.name();
		}

		if ( aProtocol == null && aStoreDescriptor != null ) {
			// LOGGER.info( "You did not provide a protocol such as <" +
			// DEFAULT_SSL_PROTOCOL + "> or <" + Scheme.HTTPS.toProtocol() + ">,
			// falling back <" + DEFAULT_SSL_PROTOCOL + ">." );
			aProtocol = TransportLayerProtocol.TLS.name();
		}

		if ( aStoreDescriptor == null ) aStoreDescriptor = getKeyStoreDescriptor();
		if ( aMaxConnections < 0 ) aMaxConnections = getMaxConnections();

		// HTTP |-->
		if ( (theScheme == null && aProtocol == null) || theScheme == Scheme.HTTP ) {
			try {
				HttpServer theHttpServer = HttpServer.create();
				theHttpServer.bind( new InetSocketAddress( aPort ), aMaxConnections );
				open( theHttpServer );
			}
			catch ( IOException e ) {
				throw new OpenException( "Unable to bind to port <" + aPort + ">: " + ExceptionUtility.toMessage( e ), e );
			}
		}
		// HTTP <--|
		// HTTPS |-->
		else {
			try {
				InetSocketAddress theAddress = new InetSocketAddress( aPort );
				SSLContext theSSLContext = SSLContext.getInstance( aProtocol ); // "TLS"
				KeyStore theKeyStore = KeyStore.getInstance( aStoreDescriptor.getStoreType().name() ); // "JKS"
				FileInputStream theKeystoreInputStream = new FileInputStream( aStoreDescriptor.getStoreFile() );
				theKeyStore.load( theKeystoreInputStream, aStoreDescriptor.getStorePassword().toCharArray() );
				KeyManagerFactory theKeyManagerFactory = KeyManagerFactory.getInstance( KeyManagerFactory.getDefaultAlgorithm() );
				theKeyManagerFactory.init( theKeyStore, aStoreDescriptor.getKeyPassword().toCharArray() );
				TrustManagerFactory theTrustManagerFactory = TrustManagerFactory.getInstance( TrustManagerFactory.getDefaultAlgorithm() );
				theTrustManagerFactory.init( theKeyStore );
				theSSLContext.init( theKeyManagerFactory.getKeyManagers(), theTrustManagerFactory.getTrustManagers(), null );
				HttpsServer theHttpsServer = HttpsServer.create( theAddress, aMaxConnections );
				theHttpsServer.setHttpsConfigurator( new HttpsRestConfigurator( theSSLContext ) );
				theHttpsServer.setExecutor( null ); // creates default executor
				open( theHttpsServer );
			}
			catch ( IOException | NoSuchAlgorithmException | KeyStoreException | CertificateException | UnrecoverableKeyException | KeyManagementException e ) {
				throw new OpenException( "Unable to bind to port <" + aPort + ">: " + ExceptionUtility.toMessage( e ), e );
			}
		}
		// HTTPS <--|
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ConnectionStatus getConnectionStatus() {
		return _connectionStatus;
	}

	// /////////////////////////////////////////////////////////////////////////
	// MEHTODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public HttpRestServer onConnectionRequest( HttpsConnectionRequestObserver aObserver ) {
		_httpsConnectionRequestObserver = aObserver;
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public HttpRestServer onBasicAuthRequest( BasicAuthObserver aBasicAuthObserver ) {
		HttpContext theHttpContext = _httpContext;
		HttpBasicAuthenticator theHttpBasicAuthenticator = null;
		if ( aBasicAuthObserver != null ) {
			theHttpBasicAuthenticator = new HttpBasicAuthenticator( aBasicAuthObserver );
			if ( theHttpContext != null ) {
				theHttpContext.setAuthenticator( theHttpBasicAuthenticator );
			}
		}
		_httpBasicAuthenticator = theHttpBasicAuthenticator;
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public HttpRestServer withRealm( String aRealm ) {
		setRealm( aRealm );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPort( int aPort ) {
		_port = aPort;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getPort() {
		return _port;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setScheme( Scheme aScheme ) {
		_scheme = aScheme;
		_protocol = null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Scheme getScheme() {
		return _scheme;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toProtocol() {
		if ( _scheme != null ) {
			return _scheme.toProtocol();
		}
		return _protocol;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setProtocol( String aProtocol ) {
		Scheme theScheme = Scheme.fromProtocol( aProtocol );
		if ( theScheme != null ) {
			_scheme = theScheme;
			_protocol = null;
		}
		else {
			_protocol = aProtocol;

			_scheme = null;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public KeyStoreDescriptor getKeyStoreDescriptor() {
		return _keyStoreDescriptor;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setKeyStoreDescriptor( KeyStoreDescriptor aStoreDescriptor ) {
		_keyStoreDescriptor = aStoreDescriptor;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getMaxConnections() {
		return _maxConnections;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMaxConnections( int aMaxConnections ) {
		_maxConnections = aMaxConnections;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Gets the http server.
	 *
	 * @return the http server
	 */
	protected HttpServer getHttpServer() {
		return _httpServer;
	}

	/**
	 * A hook to be used when using custom {@link HttpServer} (
	 * {@link HttpsServer}) by custom open(...) methods of sub-classes of this
	 * {@link HttpRestServerImpl}. E.g {@link HttpRestServerImpl} uses this hook
	 * to pre-configure a {@link HttpsServer} for HTTPS.
	 * 
	 * The passed {@link HttpServer} ( {@link HttpsServer}) must already be
	 * bound to a port and enabled with the number of concurrent connections as
	 * of {@link HttpServer#bind(InetSocketAddress, int)}.
	 * 
	 * @param aHttpServer The {@link HttpServer} to be used. E.g. an
	 *        {@link HttpsServer} might be used to enable HTTPS.
	 * 
	 * @throws IOException in case opening with the provided
	 *         {@link HttpRestServer} fails.
	 */
	protected void open( HttpServer aHttpServer ) throws IOException {
		if ( _connectionStatus == ConnectionStatus.OPENED ) {
			throw new OpenException( "Connection is still in status <" + _connectionStatus + ">. Close the connection before reopening!" );
		}
		if ( _executorService != null ) aHttpServer.setExecutor( _executorService );
		HttpContext theHttpContext = aHttpServer.createContext( CONTEXT_PATH, new EndpointHttpHandler() );
		HttpBasicAuthenticator theHttpBasicAuthenticator = _httpBasicAuthenticator;
		if ( theHttpBasicAuthenticator != null ) {
			theHttpContext.setAuthenticator( theHttpBasicAuthenticator );
		}
		aHttpServer.start();
		_httpServer = aHttpServer;
		_httpContext = theHttpContext;
		_connectionStatus = ConnectionStatus.OPENED;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Handles the {@link Result} in case of authorization failure.
	 * 
	 * @param aHttpExchange The {@link com.sun.net.httpserver.HttpExchange}
	 *        which's "WWW-Authenticate" header is to be modified.
	 * 
	 * @return the {@link Result} of type {@link Authenticator.Failure}.
	 */
	private Authenticator.Result toBasicAuthFailure( HttpExchange aHttpExchange ) {
		return new Authenticator.Failure( HttpStatusCode.UNAUTHORIZED.getStatusCode() );
	}

	/**
	 * Handles the {@link Result} in case of authorization success.
	 * 
	 * 
	 * 
	 * @param aHttpExchange The {@link com.sun.net.httpserver.HttpExchange}
	 *        which's "WWW-Authenticate" header is to be modified.
	 * 
	 * @return the {@link Result} of type {@link Authenticator.Success}.
	 */
	private Authenticator.Success toBasicOutSuccess( String aIdentity ) {
		return new Authenticator.Success( new HttpPrincipal( aIdentity, getRealm() ) );
	}

	/**
	 * Handles the {@link Result} in case of authorization required.
	 * 
	 * @param aHttpExchange The {@link com.sun.net.httpserver.HttpExchange}
	 *        which's "WWW-Authenticate" header is to be modified.
	 * 
	 * @return the {@link Result} of type {@link Retry}.
	 */
	private Authenticator.Result toBasicAuthRequired( HttpExchange aHttpExchange ) {
		Headers theHeaders = aHttpExchange.getResponseHeaders();
		doBasicAuthRequired( theHeaders );
		return new Authenticator.Retry( HttpStatusCode.UNAUTHORIZED.getStatusCode() );
	}

	/**
	 * Sets Basic-Authentication required for the
	 * {@link HeaderField#WWW_AUTHENTICATE}.
	 * 
	 * @param aHeaders The headers which to modify accordingly.
	 */
	private void doBasicAuthRequired( Headers aHeaders ) {
		aHeaders.set( HeaderField.WWW_AUTHENTICATE.getName(), HeaderFields.BASIC_REALM + "=\"" + getRealm() + "\"" );
	}

	///////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	///////////////////////////////////////////////////////////////////////////

	/**
	 * Extension of the {@link com.sun.net.httpserver.Authenticator} for doing
	 * custom HTTP Basic-Authentication.
	 */
	private class HttpBasicAuthenticator extends Authenticator {

		private BasicAuthObserver _basicAuthObserver;

		/**
		 * Instantiates a new http basic authenticator.
		 *
		 * @param aBasicAuthObserver the basic auth observer
		 */
		public HttpBasicAuthenticator( BasicAuthObserver aBasicAuthObserver ) {
			_basicAuthObserver = aBasicAuthObserver;
		}

		/**
		 * Authenticate.
		 *
		 * @param aHttpExchange The {@link HttpExchange} doing the "physical"
		 *        HTTP-Response.
		 * @return the result
		 */
		@Override
		public Result authenticate( HttpExchange aHttpExchange ) {
			Headers theRequestHeaders = aHttpExchange.getRequestHeaders();
			String theAuthHeader = theRequestHeaders.getFirst( HeaderField.AUTHORIZATION.getName() );
			BasicAuthCredentials theCredentials = null;
			if ( theAuthHeader == null ) {
				BasicAuthResponse theBasicAuthResponse = _basicAuthObserver.onBasicAuthRequest( aHttpExchange.getLocalAddress(), aHttpExchange.getRemoteAddress(), HttpMethod.fromHttpMethod( aHttpExchange.getRequestMethod() ), aHttpExchange.getRequestURI().getPath(), theCredentials, getRealm() );
				if ( theBasicAuthResponse != BasicAuthResponse.BASIC_AUTH_SUCCESS ) {
					return toBasicAuthRequired( aHttpExchange );
				}
				return new Authenticator.Success( new HttpPrincipal( ANONYMOUS, getRealm() ) );
			}
			else {
				int theMarker = theAuthHeader.indexOf( ' ' );
				if ( theMarker == -1 || !theAuthHeader.substring( 0, theMarker ).equals( AuthType.BASIC.getName() ) ) {
					return toBasicAuthFailure( aHttpExchange );
				}
				byte[] theCredentialChars = Base64.getDecoder().decode( theAuthHeader.substring( theMarker + 1 ) );
				String theCredentialsText = new String( theCredentialChars );
				theMarker = theCredentialsText.indexOf( ':' );
				theCredentials = new BasicAuthCredentialsImpl( theCredentialsText.substring( 0, theMarker ), theCredentialsText.substring( theMarker + 1 ) );
			}
			BasicAuthResponse theBasicAuthResponse = _basicAuthObserver.onBasicAuthRequest( aHttpExchange.getLocalAddress(), aHttpExchange.getRemoteAddress(), HttpMethod.fromHttpMethod( aHttpExchange.getRequestMethod() ), aHttpExchange.getRequestURI().getPath(), theCredentials, getRealm() );
			if ( theBasicAuthResponse == null ) {
				throw new NullPointerException( "Your <HttpBasicAuthenticator> instance must return an element of type <BasicAuthResponse> and not null." );
			}
			switch ( theBasicAuthResponse ) {
			case BASIC_AUTH_SUCCESS:
				return toBasicOutSuccess( theCredentials.getIdentity() );
			case BASIC_AUTH_REQUIRED:
				return toBasicAuthRequired( aHttpExchange );
			case BASIC_AUTH_FAILURE:
				return toBasicAuthFailure( aHttpExchange );
			default:
				throw new BugException( "Missing case statement for <" + theBasicAuthResponse + "> in implementation!" );
			}
		}
	}

	/**
	 * Main {@link com.sun.net.httpserver.HttpHandler} managing the dispatch of
	 * incoming requests to the registered {@link RestRequestObserver} instances
	 * depending on the according {@link RestEndpoint}'s Locator-Pattern, HTTP
	 * method and so on.
	 */
	private class EndpointHttpHandler implements HttpHandler {

		/**
		 * Handle.
		 *
		 * @param aHttpExchange The {@link HttpExchange} doing the "physical"
		 *        HTTP-Response.
		 * @throws IOException Signals that an I/O exception has occurred.
		 */
		@Override
		public void handle( HttpExchange aHttpExchange ) throws IOException {
			HttpMethod theHttpMethod = HttpMethod.fromHttpMethod( aHttpExchange.getRequestMethod() );
			if ( theHttpMethod != null ) {
				URI theRequestURI = aHttpExchange.getRequestURI();
				HttpServerResponse theHttpServerResponse = new HttpServerResponseImpl( HttpRestServerImpl.this );
				InetSocketAddress theLocalAddress = aHttpExchange.getLocalAddress();
				InetSocketAddress theRemoteAddress = aHttpExchange.getRemoteAddress();
				Headers theRequestHeaders = aHttpExchange.getRequestHeaders();
				RequestHeaderFields theRequestHeaderFields = new RequestHeaderFieldsImpl( theRequestHeaders );
				UrlBuilder theUrl = new UrlBuilderImpl( theRequestURI.toString() );
				if ( theUrl.getScheme() == null ) {
					Scheme theScheme = _scheme;
					if ( theScheme == null ) {
						theScheme = TransportLayerProtocol.toScheme( _protocol );
					}
					theUrl.setScheme( theScheme );
				}
				if ( theUrl.getHost() == null ) {
					theUrl.setHost( Literal.LOCALHOST.getName() );
				}
				if ( theUrl.getPort() == -1 ) {
					theUrl.setPort( _port );
				}
				try {
					onHttpRequest( aHttpExchange, theLocalAddress, theRemoteAddress, theHttpMethod, theUrl, theRequestHeaderFields, aHttpExchange.getRequestBody(), theHttpServerResponse );
				}
				catch ( BasicAuthRequiredException e ) {
					doBasicAuthRequired( aHttpExchange.getResponseHeaders() );
					LOGGER.info( "Required HTTP Basic-Authentication with status <" + e.getStatusCode() + "> with code <" + e.getStatusCode().getStatusCode() + "> for request URL <" + theRequestURI + "> with request method <" + aHttpExchange.getRequestMethod() + ">: " + ExceptionUtility.toMessage( e ) );
					onHttpException( aHttpExchange, theLocalAddress, theRemoteAddress, theHttpMethod, theUrl, theRequestHeaderFields, theHttpServerResponse, e, e.getStatusCode() );
				}
				catch ( HttpStatusException e ) {
					LOGGER.warn( "Responding status <" + e.getStatusCode() + "> with code <" + e.getStatusCode().getStatusCode() + "> for request URL <" + theRequestURI + "> with request method <" + aHttpExchange.getRequestMethod() + ">: " + ExceptionUtility.toMessage( e ) );
					onHttpException( aHttpExchange, theLocalAddress, theRemoteAddress, theHttpMethod, theUrl, theRequestHeaderFields, theHttpServerResponse, e, e.getStatusCode() );
				}
				catch ( Exception e ) {
					LOGGER.warn( "Bad request <" + e.getClass().getName() + "> for request URL <" + theRequestURI + "> with request method <" + aHttpExchange.getRequestMethod() + ">: " + ExceptionUtility.toMessage( e ), e );
					onHttpException( aHttpExchange, theLocalAddress, theRemoteAddress, theHttpMethod, theUrl, theRequestHeaderFields, theHttpServerResponse, e, HttpStatusCode.BAD_REQUEST );
				}
			}
			else {
				LOGGER.warn( "Unknown HTTP-Method <" + aHttpExchange.getRequestMethod() + "> when querying resource locator <" + aHttpExchange.getLocalAddress() + ">." );
				aHttpExchange.sendResponseHeaders( HttpStatusCode.INTERNAL_SERVER_ERROR.getStatusCode(), NO_RESPONSE_BODY );
			}
		}

		/**
		 * Invokes the registered {@link RestEndpoint} instances upon an
		 * incoming HTTP-Request.
		 * 
		 * @param aHttpExchange The {@link HttpExchange} doing the "physical"
		 *        HTTP-Response.
		 * @param aLocalAddress The host and port of your REST service.
		 * @param aRemoteAddress The host and port for the caller.
		 * @param aHttpMethod The {@link HttpMethod} of the request.
		 * @param aUrl The {@link Url} from which to take the URL specific data.
		 * @param aRequestHeaderFields The Header-Fields ({@link HeaderFields})
		 *        belonging to the request.
		 * @param aHttpInputStream The body passed by the request.
		 * @param aHttpServerResponse The {@link HttpServerResponse} to be
		 *        worked with.
		 * 
		 * @return A {@link HttpServerResponse} instance to by used by the
		 *         extension to produce an according HTTP-Response.
		 * @throws HttpStatusException thrown in case of an {@link RestEndpoint}
		 *         responsible for the given request encountered a problem or
		 *         none {@link RestEndpoint} felt responsible to produce a
		 *         {@link HttpServerResponse}.
		 * @throws IOException Thrown in case even the {@link HttpExchange}
		 *         failed to process.
		 * @throws MarshalException Thrown when marshaling / serializing the
		 *         response failed.
		 * @throws UnsupportedMediaTypeException Thrown in case a Media-Type has
		 *         been provided which is not supported.
		 */
		private void onHttpRequest( HttpExchange aHttpExchange, InetSocketAddress aLocalAddress, InetSocketAddress aRemoteAddress, HttpMethod aHttpMethod, UrlBuilder aUrl, RequestHeaderFields aRequestHeaderFields, InputStream aHttpInputStream, HttpServerResponse aHttpServerResponse ) throws HttpStatusException, MarshalException, IOException {
			HttpRestServerImpl.super.onHttpRequest( aLocalAddress, aRemoteAddress, aHttpMethod, aUrl, aRequestHeaderFields, aHttpInputStream, aHttpServerResponse );
			HttpStatusCode theHttpStatusCode = aHttpServerResponse.getHttpStatusCode();
			if ( theHttpStatusCode == null ) theHttpStatusCode = HttpStatusCode.OK;
			doHttpResponse( aHttpExchange, aRequestHeaderFields, aHttpServerResponse, theHttpStatusCode );
		}

		/**
		 * Invokes the registered {@link HttpExceptionHandler} instance (if any)
		 * upon an Exception whilst processing an incoming HTTP-Request.
		 * 
		 * @param aHttpExchange The {@link HttpExchange} doing the "physical"
		 *        HTTP-Response.
		 * @param aLocalAddress The host and port of your REST service.
		 * @param aRemoteAddress The host and port for the caller.
		 * @param aHttpMethod The {@link HttpMethod} of the request.
		 * @param aUrl The {@link Url} from which to take the URL specific data.
		 * @param aRequestHeaderFields The Header-Fields ({@link HeaderFields})
		 *        belonging to the request.
		 * @param aHttpInputStream The body passed by the request.
		 * @param aException The cause of the problems.
		 * @param aHttpServerResponse The {@link HttpServerResponse} to be
		 *        worked with
		 * @param aHttpStatusCode The status code to be set for the response.
		 * 
		 * @return A {@link HttpServerResponse} instance to by used by the
		 *         extension to produce an according HTTP-Response.
		 * 
		 * @throws HttpStatusException thrown in case of an {@link RestEndpoint}
		 *         responsible for the given request encountered a problem or
		 *         none {@link RestEndpoint} felt responsible to produce a
		 *         {@link HttpServerResponse}.
		 * @throws IOException Thrown in case even the {@link HttpExchange}
		 *         failed to process.
		 * @throws MarshalException Thrown when marshaling / serializing the
		 *         response failed.
		 * @throws UnsupportedMediaTypeException Thrown in case a Media-Type has
		 *         been provided which is not supported.
		 */
		protected void onHttpException( HttpExchange aHttpExchange, InetSocketAddress aLocalAddress, InetSocketAddress aRemoteAddress, HttpMethod aHttpMethod, UrlBuilder aUrl, RequestHeaderFields aRequestHeaderFields, HttpServerResponse aHttpServerResponse, Exception aException, HttpStatusCode aHttpStatusCode ) throws IOException {
			aHttpServerResponse.setHttpStatusCode( aHttpStatusCode );
			if ( _httpExceptionHandler != null ) {
				try {
					RestRequestEvent theRestRequestEvent = new RestRequestEventImpl( aLocalAddress, aRemoteAddress, aHttpMethod, aUrl, null, aRequestHeaderFields, aHttpExchange.getRequestBody(), HttpRestServerImpl.this );
					_httpExceptionHandler.onHttpError( theRestRequestEvent, aHttpServerResponse, aException, aHttpStatusCode );
					finalizeMediaType( aHttpServerResponse );
					doHttpResponse( aHttpExchange, aRequestHeaderFields, aHttpServerResponse, aHttpStatusCode );
				}
				catch ( Exception failed ) {
					finalizeHttpExceptionResponse( aHttpExchange, aException, aHttpStatusCode );
				}
			}
			else {
				finalizeHttpExceptionResponse( aHttpExchange, aException, aHttpStatusCode );
			}
		}

		/**
		 * If an unsupported Media-Type is set, then the first supported
		 * Media-Type is set. We assume that we already have (if any) taken over
		 * the Media-Type from the according request. Now we finalize for
		 * erroneous situations to make sure we support the Media-Type
		 * 
		 * @param aHttpServerResponse The {@link HttpServerResponse} for which
		 *        to finalize the Media-Type.
		 */
		protected void finalizeMediaType( HttpServerResponse aHttpServerResponse ) {
			if ( aHttpServerResponse.getResponse() != null ) {
				ContentType theContentType = aHttpServerResponse.getHeaderFields().getContentType();
				if ( theContentType == null || !HttpRestServerImpl.this.hasMediaTypeFactory( theContentType.getMediaType() ) ) {
					String theTypeText = theContentType != null ? theContentType.toHttpMediaType() : null;
					LOGGER.warn( "Unsupported Content-Type <" + theTypeText + "> detected in HTTP-Server-Response, trying fallback ..." );
					if ( HttpRestServerImpl.this.getFactoryMediaTypes().length == 0 ) {
						LOGGER.warn( "No fallback Content-Type detected for HTTP-Server-Response as no Media-Type-Factories have been configured ..." );
					}
					else if ( HttpRestServerImpl.this.getFactoryMediaTypes().length != 0 ) {
						MediaType theMediaType = HttpRestServerImpl.this.getFactoryMediaTypes()[0];
						LOGGER.info( "Using fallback Content-Type <" + theMediaType.toHttpMediaType() + "> for HTTP-Server-Response ..." );
						aHttpServerResponse.getHeaderFields().putContentType( theMediaType );
					}
				}
			}
		}

		/**
		 * Does the HTTP-Response handling such as content negotiation, body
		 * construction and status code processing.
		 * 
		 * @param aHttpExchange The {@link HttpExchange} doing the "physical"
		 *        HTTP-Response.
		 * @param aRequestHeaderFields The Header-Fields of the original
		 *        request.
		 * @param aHttpServerResponse The constructed {@link HttpServerResponse}
		 *        to be sent back to the client.
		 * @param aHttpStatusCode The status code to be set for the response.
		 * 
		 * @throws IOException Thrown in case even the {@link HttpExchange}
		 *         failed to process.
		 * @throws MarshalException Thrown when marshaling / serializing the
		 *         response failed.
		 * @throws UnsupportedMediaTypeException Thrown in case a Media-Type has
		 *         been provided which is not supported.
		 */
		protected void doHttpResponse( HttpExchange aHttpExchange, RequestHeaderFields aRequestHeaderFields, HttpServerResponse aHttpServerResponse, HttpStatusCode aHttpStatusCode ) throws IOException, MarshalException, UnsupportedMediaTypeException {
			Headers theResponseHeaders = aHttpExchange.getResponseHeaders();
			for ( String eKey : aHttpServerResponse.getHeaderFields().keySet() ) {
				theResponseHeaders.put( eKey, aHttpServerResponse.getHeaderFields().get( eKey ) );
			}
			Object theResponse = aHttpServerResponse.getResponse();
			if ( (theResponse instanceof InputStream) ) {
				aHttpExchange.sendResponseHeaders( aHttpStatusCode.getStatusCode(), CHUNCKED_ENCODING );
				InputStream theInputStream = (InputStream) theResponse;
				HttpRestClientImpl.pipe( theInputStream, aHttpExchange.getResponseBody() );
				aHttpExchange.getResponseBody().flush();
			}
			else {
				byte[] theBytes = null;
				if ( theResponse != null ) {
					try {
						theBytes = aHttpServerResponse.toHttpBody().getBytes();
					}
					catch ( BadResponseException e ) {
						LOGGER.warn( e.getMessage() + ": Trying fallback procedure ..." );
						theBytes = toResponseBody( aHttpServerResponse.getResponse(), aRequestHeaderFields, aHttpServerResponse.getHeaderFields() );
					}
				}
				if ( theBytes != null && theBytes.length != 0 ) {
					aHttpExchange.sendResponseHeaders( aHttpStatusCode.getStatusCode(), theBytes.length );
					aHttpExchange.getResponseBody().write( theBytes );
					aHttpExchange.getResponseBody().flush();
				}
				else {
					aHttpExchange.sendResponseHeaders( aHttpStatusCode.getStatusCode(), NO_RESPONSE_BODY );
				}
			}
			aHttpExchange.getResponseBody().close();
		}

		/**
		 * If nothing more is possible, finish with minimum means to get
		 * response through to the client.
		 * 
		 * @param aHttpExchange The {@link HttpExchange} doing the "physical"
		 *        HTTP-Response.
		 * @param aException The cause of the problems.
		 * @param aHttpStatusCode The status code to be set for the
		 *        HTTP-Response.
		 * 
		 * @throws IOException Thrown in case even the {@link HttpExchange}
		 *         failed to process.
		 */
		protected void finalizeHttpExceptionResponse( HttpExchange aHttpExchange, Exception aException, HttpStatusCode aHttpStatusCode ) throws IOException {
			LOGGER.warn( "Unable fully satisfy <" + getHttpExceptionHandling() + "> mode, falling back to <" + HttpExceptionHandling.EMPTY + "> : " + aException.getMessage(), aException );
			aHttpExchange.sendResponseHeaders( aHttpStatusCode.getStatusCode(), NO_RESPONSE_BODY );
			aHttpExchange.getResponseBody().close();
		}

	}

	private class DefaultErrorHandler implements HttpExceptionHandler {

		@Override
		public void onHttpError( RestRequestEvent aRequestEvent, HttpServerResponse aHttpServerResponse, Exception aException, HttpStatusCode aHttpStatusCode ) {
			HttpBodyMap theBodyMap;
			switch ( getHttpExceptionHandling() ) {
			case EMPTY:
				aHttpServerResponse.setResponse( null );
				aHttpServerResponse.getHeaderFields().clear();
				break;
			case KEEP:
				break;
			case MERGE:
				theBodyMap = toResponseBodyMap( aHttpServerResponse );
				if ( !theBodyMap.hasStatusCode() ) {
					theBodyMap.putStatusCode( aHttpStatusCode );
				}
				if ( !theBodyMap.hasStatusAlias() ) {
					theBodyMap.putStatusAlias( aHttpStatusCode );
				}
				if ( !theBodyMap.hasStatusException() ) {
					theBodyMap.putStatusException( aException );
				}
				if ( !theBodyMap.hasStatusMessage() && aException.getMessage() != null && aException.getMessage().length() > 0 ) {
					theBodyMap.putStatusMessage( aException.getMessage() );
				}
				if ( !theBodyMap.hasStatusTimeStamp() ) {
					theBodyMap.putStatusTimeStamp();
				}
				aHttpServerResponse.setResponse( theBodyMap );
				break;
			case UPDATE:
				theBodyMap = toResponseBodyMap( aHttpServerResponse );
				theBodyMap.putStatusCode( aHttpStatusCode );
				theBodyMap.putStatusAlias( aHttpStatusCode );
				theBodyMap.putStatusException( aException );
				theBodyMap.putStatusMessage( aException.getMessage() );
				theBodyMap.putStatusTimeStamp();
				aHttpServerResponse.setResponse( theBodyMap );
				break;
			case REPLACE:
				theBodyMap = new HttpBodyMapImpl();
				theBodyMap.putStatusCode( aHttpStatusCode );
				theBodyMap.putStatusAlias( aHttpStatusCode );
				theBodyMap.putStatusException( aException );
				theBodyMap.putStatusMessage( aException.getMessage() );
				theBodyMap.putStatusTimeStamp();
				aHttpServerResponse.setResponse( theBodyMap );
				break;
			default:
				throw new BugException( "Missing case statement for <" + getHttpExceptionHandling() + "> in implementation!" );
			}
		}

		/**
		 * Creates a {@link HttpBodyMap} from the provided HTTP-Response.
		 * Usually used in erroneous situations. In case
		 * {@link HttpServerResponse#getResponse()} returns a
		 * {@link HttpBodyMap}, then this is returned. In case
		 * {@link HttpServerResponse#getResponse()} returns an
		 * {@link InputStream}, then a new {@link HttpBodyMap} is returned: This
		 * is to prevent providing the stream's data in an exceptional state.
		 * 
		 * @param aHttpServerResponse The {@link HttpServerResponse} from which
		 *        to extract the response representing {@link HttpBodyMap}.
		 * 
		 * @return The {@link HttpBodyMap} representation of the HTTP-Response's
		 *         response object.
		 */
		protected HttpBodyMap toResponseBodyMap( HttpServerResponse aHttpServerResponse ) {
			HttpBodyMap theHttpBodyMap = null;
			Object theResponse = aHttpServerResponse.getResponse();
			if ( theResponse != null ) {
				if ( theResponse instanceof HttpBodyMap ) {
					theHttpBodyMap = (HttpBodyMap) theResponse;
				}
				else if ( !(theResponse instanceof InputStream) ) {
					theHttpBodyMap = new HttpBodyMapImpl( aHttpServerResponse.getResponse() );
				}
			}
			if ( theHttpBodyMap == null ) theHttpBodyMap = new HttpBodyMapImpl();
			return theHttpBodyMap;
		}

	}

	/**
	 * This class is used to configure the HTTPS parameters for each incoming
	 * HTTPS connection in order to change the default configuration:.
	 */
	private class HttpsRestConfigurator extends HttpsConfigurator {

		/**
		 * {@inheritDoc}
		 */
		public HttpsRestConfigurator( SSLContext aSSLContext ) {
			super( aSSLContext );
		}

		/**
		 * TODO: Not sure on how to proceed with
		 * {@link HttpsParameters#setNeedClientAuth(boolean)}. Do we need to
		 * propagate such a flag back from
		 * {@link HttpsConnectionRequestObserver} to the
		 * {@link HttpsParameters#setNeedClientAuth(boolean)}?
		 *
		 * @param aHttpsParams the the https params
		 */
		@Override
		public void configure( HttpsParameters aHttpsParams ) {
			HttpsConnectionRequestObserver theObserver = _httpsConnectionRequestObserver;
			if ( theObserver != null ) {
				InetSocketAddress theRemoteAddress = aHttpsParams.getClientAddress();
				InetSocketAddress theLocalAddress = null;
				HttpServer theServer = getHttpServer();
				if ( theServer != null ) {
					theLocalAddress = theServer.getAddress();
				}
				else {
					LOGGER.warn( "Unable to determine the local address for remote address <" + theRemoteAddress.toString() + ">, the server might have been closed in the meantimne." );
				}
				SSLContext theSSLContext = getSSLContext();
				SSLParameters theSSLParams = theSSLContext.getDefaultSSLParameters();
				theObserver.onHttpsConnectionRequest( theLocalAddress, theRemoteAddress, theSSLParams );
				aHttpsParams.setSSLParameters( theSSLParams );
			}
		}
	}
}
