// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;

import org.refcodes.collection.Attribute;
import org.refcodes.collection.impls.AttributeImpl;

public final class ReflectionUtility {

	public static final String ALIAS_SET = "set";
	public static final String ALIAS_GET = "get";
	public static final String ALIAS_HAS = "has";
	public static final String ALIAS_IS = "is";
	public static final String ALIAS_TOSTRING = "toString";

	public static final NonExistingValueClass NON_EXISTING_VALUE = new NonExistingValueClass();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Private empty constructor to prevent instantiation as of being a utility
	 * with just static public methods.
	 */
	private ReflectionUtility() {}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * This method uses reflection on order to analyze a given object. The java
	 * beans attributes and their values are retrieved and returned in an array
	 * of name-to-value pairs.
	 * 
	 * @param anObject The object to be analyzed.
	 * @return An array of objects containing the java beans name-to-value
	 *         pairs.
	 */
	public static Attribute[] toBeanAttributes( Object anObject ) {
		String eMethodName;
		boolean isBeanMethod;
		Object eInvokeReturn;
		ArrayList<Attribute> theAttributeValueStructList = new ArrayList<Attribute>();
		Method[] theMethodArray = anObject.getClass().getMethods();

		try {
			// ------------------------
			// Look at all the methods:
			// ------------------------
			for ( int i = 0; i < theMethodArray.length; i++ )
				// --------------------------------------------------------
				// Look at the signature: 0 arguments, return type != void:
				// --------------------------------------------------------
				if ( (theMethodArray[i].getParameterTypes().length == 0) && (!theMethodArray[i].getReturnType().equals( Void.TYPE )) ) {
					isBeanMethod = false;
					eInvokeReturn = NON_EXISTING_VALUE;
					eMethodName = theMethodArray[i].getName();

					// --------------
					// Public method?
					// --------------
					if ( (theMethodArray[i].getModifiers() == Modifier.PUBLIC) || (theMethodArray[i].getModifiers() == 9) ) {

						// ---------------
						// Getter methods?
						// ---------------
						if ( (eMethodName.indexOf( ALIAS_GET ) == 0) && Character.isUpperCase( eMethodName.charAt( ALIAS_GET.length() ) ) ) {
							isBeanMethod = true;
							eMethodName = eMethodName.substring( ALIAS_GET.length(), eMethodName.length() );
						}
						else if ( (eMethodName.indexOf( ALIAS_HAS ) == 0) && Character.isUpperCase( eMethodName.charAt( ALIAS_HAS.length() ) ) ) {
							isBeanMethod = true;
							eMethodName = eMethodName.substring( ALIAS_HAS.length(), eMethodName.length() );
						}
						else if ( (eMethodName.indexOf( ALIAS_IS ) == 0) && Character.isUpperCase( eMethodName.charAt( ALIAS_IS.length() ) ) ) {
							isBeanMethod = true;
							eMethodName = eMethodName.substring( ALIAS_IS.length(), eMethodName.length() );
						}
						else if ( eMethodName.equals( ALIAS_TOSTRING ) ) isBeanMethod = true;

						// ------------------
						// Bean method found?
						// ------------------
						if ( isBeanMethod ) {

							try {
								// ------------------
								// Invoke the method:
								// ------------------
								eInvokeReturn = theMethodArray[i].invoke( anObject, new Object[] {} );
							}
							catch ( Exception e ) {
								System.err.println( "ReflectionUtility.setAttribute(): Caught an exception of type <" + e.getClass().getName() + "> while invoking the method <" + eMethodName + "> with message = " + e.getMessage() + "." );
								e.printStackTrace();
							}

							theAttributeValueStructList.add( new AttributeImpl( eMethodName, eInvokeReturn ) );
						}
					}
				}

		}
		catch ( Exception exc ) {
			System.err.println( "ReflectionUtility.setAttribute(): Caught an exception of type <" + exc.getClass().getName() + "> with message = " + exc.getMessage() + "." );
			exc.printStackTrace();
		}

		return theAttributeValueStructList.toArray( new AttributeImpl[theAttributeValueStructList.size()] );
	}

	/**
	 * This method sets a java beans attribute for the given object. The
	 * corresponding java beans method must begin with a 'set', the following
	 * letter must be in upper case and it must only take one argument being of
	 * the same (or super-) type as the attribute's type.
	 * 
	 * @param anObject The object which's java beans method is to be called.
	 * @param anAttributeName The attribute name of the java beans method.
	 * @param anAttributeValue The value of the attribute to be set.
	 * @exception NoSuchMethodException Description of the Exception
	 */
	public static void setAttribute( Object anObject, String anAttributeName, Object anAttributeValue ) throws NoSuchMethodException {

		String theMethodName;
		boolean isCalled = false;
		Method[] theMethodArray = anObject.getClass().getMethods();

		try {
			// ------------------------
			// Look at all the methods:
			// ------------------------
			for ( int i = 0; i < theMethodArray.length; i++ )

				// -------------------------------------------------------
				// Look at the signature: 1 argument, return type == void:
				// -------------------------------------------------------
				if ( (theMethodArray[i].getParameterTypes().length == 1) && (theMethodArray[i].getParameterTypes()[0].isAssignableFrom( anAttributeValue.getClass() )) && (theMethodArray[i].getReturnType().equals( Void.TYPE )) ) {
					theMethodName = theMethodArray[i].getName();

					if ( (theMethodArray[i].getModifiers() == Modifier.PUBLIC) || (theMethodArray[i].getModifiers() == 9) ) {
						// --------------------
						// Find setter methods:
						// --------------------
						if ( (theMethodName.indexOf( ALIAS_SET ) == 0) && Character.isUpperCase( theMethodName.charAt( ALIAS_SET.length() ) ) ) {
							theMethodName = theMethodName.substring( ALIAS_SET.length(), theMethodName.length() );
							if ( anAttributeName.equalsIgnoreCase( theMethodName ) ) {
								try {
									// ----------------
									// Call the method:
									// ----------------
									theMethodArray[i].invoke( anObject, new Object[] {
										anAttributeValue
									} );
									isCalled = true;
								}
								catch ( Exception e ) {
									System.err.println( "ReflectionUtility.setAttribute(): Caught an exception of type <" + e.getClass().getName() + "> while invoking the method <" + theMethodName + "> with message = " + e.getMessage() + "." );
									e.printStackTrace();
									break;
								}
								break;
							}
						}
					}
				}
		}
		catch ( Exception exc ) {
			System.err.println( "ReflectionUtility.setAttribute(): Caught an exception of type <" + exc.getClass().getName() + "> with message = " + exc.getMessage() + "." );
			exc.printStackTrace();
		}

		if ( !isCalled ) throw new NoSuchMethodException( "ReflectionUtility.setAttribute(): The method for the attribute <" + anAttributeName + "> for the class <" + anObject.getClass().getName() + "> cannot be found." );
	}

	/**
	 * This method sets a java beans attribute for the given object.
	 * 
	 * @param anObject The object which's attribute is to be set.
	 * @param anAttributeValueStruct The name-to-value struct for the attrubute
	 *        to be set.
	 * @exception NoSuchMethodException Description of the Exception
	 */
	public static void setAttribute( Object anObject, Attribute anAttributeValueStruct ) throws NoSuchMethodException {
		setAttribute( anObject, anAttributeValueStruct.getKey(), anAttributeValueStruct.getValue() );
	}

	/**
	 * This method tests whether the given java beans setter attribute is found
	 * for the given object
	 * 
	 * @param anObject The object which is to be tested.
	 * @param anAttributeName The attribute name.
	 * @param anAttributeType The expected (sub-) type of the attribute
	 * @return True if the object has a method with the given java beans name
	 *         which can be called with an argument of the given type.
	 */
	public static boolean hasSetterAttribute( Object anObject, String anAttributeName, Class<?> anAttributeType ) {
		String theMethodName;
		Method[] theMethodArray = anObject.getClass().getMethods();

		try {
			// ------------------------
			// Look at all the methods:
			// ------------------------
			for ( int i = 0; i < theMethodArray.length; i++ )
				// -------------------------------------------------------
				// Look at the signature: 1 argument, return type == void:
				// -------------------------------------------------------
				if ( (theMethodArray[i].getParameterTypes().length == 1) && (theMethodArray[i].getParameterTypes()[0].isAssignableFrom( anAttributeType )) && (theMethodArray[i].getReturnType().equals( Void.TYPE )) ) {

					theMethodName = theMethodArray[i].getName();

					if ( (theMethodArray[i].getModifiers() == Modifier.PUBLIC) || (theMethodArray[i].getModifiers() == 9) ) {

						// --------------------
						// Find setter methods:
						// --------------------
						if ( (theMethodName.indexOf( ALIAS_SET ) == 0) && Character.isUpperCase( theMethodName.charAt( ALIAS_SET.length() ) ) ) {
							theMethodName = theMethodName.substring( ALIAS_SET.length(), theMethodName.length() );
							// ------------------
							// Bean method found:
							// ------------------
							if ( anAttributeName.equalsIgnoreCase( theMethodName ) ) return true;
						}
					}
				}
		}
		catch ( Exception exc ) {
			System.err.println( "ReflectionUtility.setAttribute(): Caught an exception of type <" + exc.getClass().getName() + "> with message = " + exc.getMessage() + "." );
			exc.printStackTrace();
		}
		return false;
	}

	/**
	 * This method tests whether the given java beans getter attribute is found
	 * for the given object
	 * 
	 * @param anObject The object which is to be tested.
	 * @param anAttributeName The attribute name.
	 * @param aReturnType The expected (sub-) type of the return value
	 * @return True if the object has a method with the given java beans name
	 *         which returns the given (super-) type.
	 */
	public static boolean hasGetterAttribute( Object anObject, String anAttributeName, Class<?> aReturnType ) {

		String theMethodName;
		Method[] theMethodArray = anObject.getClass().getMethods();

		try {
			// ------------------------
			// Look at all the methods:
			// ------------------------
			for ( int i = 0; i < theMethodArray.length; i++ )
				// ------------------------------------------------------
				// Look at the signature: 0 arguments, given return type:
				// ------------------------------------------------------
				if ( (theMethodArray[i].getParameterTypes().length == 0) && (theMethodArray[i].getReturnType().equals( aReturnType )) ) {

					theMethodName = theMethodArray[i].getName();

					if ( (theMethodArray[i].getModifiers() == Modifier.PUBLIC) || (theMethodArray[i].getModifiers() == 9) ) {

						// -------------------
						// Find 'get' methods:
						// -------------------
						if ( (theMethodName.indexOf( ALIAS_GET ) == 0) && Character.isUpperCase( theMethodName.charAt( ALIAS_GET.length() ) ) ) {
							theMethodName = theMethodName.substring( ALIAS_GET.length(), theMethodName.length() );
							// ------------------
							// Bean method found:
							// ------------------
							if ( anAttributeName.equalsIgnoreCase( theMethodName ) ) return true;
						}
						// -------------------
						// Find 'has' methods:
						// -------------------
						else if ( (theMethodName.indexOf( ALIAS_HAS ) == 0) && Character.isUpperCase( theMethodName.charAt( ALIAS_HAS.length() ) ) ) {
							theMethodName = theMethodName.substring( ALIAS_HAS.length(), theMethodName.length() );
							// ------------------
							// Bean method found:
							// ------------------
							if ( anAttributeName.equalsIgnoreCase( theMethodName ) ) return true;
						}
						// -------------------
						// Find 'is' methods:
						// -------------------
						else if ( (theMethodName.indexOf( ALIAS_IS ) == 0) && Character.isUpperCase( theMethodName.charAt( ALIAS_IS.length() ) ) ) {
							theMethodName = theMethodName.substring( ALIAS_IS.length(), theMethodName.length() );
							// ------------------
							// Bean method found:
							// ------------------
							if ( anAttributeName.equalsIgnoreCase( theMethodName ) ) return true;
						}
						// ------------------------
						// Find 'toString' methods:
						// ------------------------
						else if ( (theMethodName.indexOf( ALIAS_TOSTRING ) == 0) && Character.isUpperCase( theMethodName.charAt( ALIAS_TOSTRING.length() ) ) ) {
							theMethodName = theMethodName.substring( ALIAS_TOSTRING.length(), theMethodName.length() );
							// ------------------
							// Bean method found:
							// ------------------
							if ( anAttributeName.equalsIgnoreCase( theMethodName ) ) return true;
						}
					}
				}
		}
		catch ( Exception exc ) {
			System.err.println( "ReflectionUtility.setAttribute(): Caught an exception of type <" + exc.getClass().getName() + "> with message = " + exc.getMessage() + "." );
			exc.printStackTrace();
		}
		return false;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * This class indicates a non existing value in the name-to-value mapping
	 */
	public static class NonExistingValueClass {}
}
