// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.InetAddress;
import java.net.UnknownHostException;

import org.refcodes.data.QueryConsts;
import org.refcodes.data.RuntimeConsts;
import org.refcodes.data.SystemConsts;

import jline.Terminal;
import jline.TerminalFactory;

/**
 * Utility for acquiring system information on the machine this process is 
 * running in.
 */
public final class SystemUtility {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Private constructor as of being utility class.
	 */
	private SystemUtility() {}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Determines the operating system your application is currently running on.
	 * 
	 * @return The {@link OperatingSystem} being detected.
	 */
	public static OperatingSystem getOperatingSystem() {
		String OS = System.getProperty( "os.name" ).toLowerCase();
		if ( OS.indexOf( "win" ) >= 0 ) { return OperatingSystem.WINDOWS; }
		if ( OS.indexOf( "mac" ) >= 0 ) { return OperatingSystem.MAC; }
		if ( OS.indexOf( "linux" ) >= 0 ) { return OperatingSystem.UNIX; }
		if ( OS.indexOf( "sunos" ) >= 0 ) { return OperatingSystem.UNIX; }
		if ( OS.indexOf( "aix" ) >= 0 ) { return OperatingSystem.UNIX; }
		if ( OS.indexOf( "nix" ) >= 0 ) { return OperatingSystem.UNIX; }
		return OperatingSystem.UNKNOWN;
	}

	/**
	 * Determines the operating system as of {@link #getOperatingSystem()} and
	 * in case a {@link OperatingSystem#WINDOWS} is being detected, then \r\n"
	 * (CRLF) is returned, else "\n" (LF) is returned.
	 * 
	 * @return The operating specific line break; on Windows it is "\r\n" (CRLF)
	 *         and on all other operating systems it is "\n" (LF).
	 */
	public static String getLineBreak() {
		String theLineBreak = System.getProperty( "line.separator" );
		if ( theLineBreak == null || theLineBreak.length() == 0 ) {
			theLineBreak = getOperatingSystem() == OperatingSystem.WINDOWS ? "\r\n" : "\n";
		}
		return theLineBreak;
	}

	/**
	 * Determines the computer's name. First it tries to get it from the
	 * {@link InetAddress}, if it fails it tries to get it from the system's
	 * environment using the {@link SystemConsts#ENV_VAR_WIN_COMPUTERNAME} (on
	 * Windows machines only) and if both fails, it returns the default
	 * {@link QueryConsts.NAME_LOCALHOST} identifier.
	 * 
	 * @return The computer's name, as fallback,
	 *         {@link SystemConsts#NAME_LOCALHOST} ("localhost") is returned.
	 */
	public static String getComputerName() {
		try {
			return InetAddress.getLocalHost().getHostName();
		}
		catch ( UnknownHostException e ) {
			String theName = System.getenv( SystemConsts.ENV_VAR_WIN_COMPUTERNAME );
			if ( theName != null && theName.length() > 0 ) { return theName; }
		}
		return SystemConsts.NAME_LOCALHOST;
	}

	/**
	 * If on a *nix alike system, this method returns the output of the
	 * "uname -a" command: "uname" prints system information, "-a" instructs it
	 * to print all information.
	 * 
	 * @return The "uname -a" output or null if "uname" is not known.
	 */
	public static String getUname() {
		return exec( "uname -a" );
	}

	/**
	 * Tries to determine the command line interpreter (CLI) - if any - this
	 * process is running in, e.g. whether we are running in a DOS console
	 * environment or a Linux alike Shell.
	 * 
	 * @return The CLI type, in case we are in a DOS / WIN_CMD console
	 *         environment, {@link CommandLineInterpreter#WIN_CMD} is returned
	 *         (for example).
	 */
	public static CommandLineInterpreter getCommandLineInterpreter() {
		if ( isCygwin() ) { return CommandLineInterpreter.SHELL; }
		boolean hasConsole = System.console() != null;
		switch ( getOperatingSystem() ) {
		case WINDOWS:
			if ( hasConsole ) { return CommandLineInterpreter.WIN_CMD; }
			break;
		case UNIX:
			if ( hasConsole ) { return CommandLineInterpreter.SHELL; }
		default:
		}
		return hasConsole ? CommandLineInterpreter.UNKNOWN : CommandLineInterpreter.NONE;
	}

	/**
	 * Determines the width in characters of the system's console in use. In
	 * case you pass a "-Dconsole.width=n" (where n stands for the number of
	 * chars per row) JVM argument, then your width is taken, else the actual
	 * console's width is being tried to be determined. In case this fails -1 is
	 * returned. In such a case you might use
	 * {@link RuntimeConsts#MAX_CONSOLE_WIDTH},
	 * {@link RuntimeConsts#NORM_CONSOLE_WIDTH} or
	 * {@link RuntimeConsts#MIN_CONSOLE_WIDTH}
	 * 
	 * @return The width of the console in characters or -1 if the width cannot
	 *         be determined.
	 */
	public static int getConsoleWidth() {
		int theConsoleWidth = TerminalFactory.get().getWidth();
		String theResult = System.getProperty( SystemConsts.SYS_PROP_CONSOLE_WIDTH );
		if ( theResult != null ) {
			try {
				theConsoleWidth = Integer.valueOf( theResult.toString() );
			}
			catch ( NumberFormatException e ) {}
		}
		theResult = System.getProperty( "COLUMNS" );
		if ( theResult != null ) {
			try {
				theConsoleWidth = Integer.valueOf( theResult.toString() );
			}
			catch ( NumberFormatException e ) {}
		}
		if ( theConsoleWidth <= 1 ) {
			theConsoleWidth = -1;
		}
		return theConsoleWidth;
	}

	/**
	 * Determines the height in characters of the system's console in use. In
	 * case you pass a "-Dconsole.height=n" (where n stands for the number of
	 * lines) JVM argument, then your height is taken, else the actual console's
	 * height is being tried to be determined. In case this fails -1 is
	 * returned. In such a case you might use
	 * {@link RuntimeConsts#MAX_CONSOLE_HEIGHT},
	 * {@link RuntimeConsts#NORM_CONSOLE_HEIGHT} or
	 * {@link RuntimeConsts#MIN_CONSOLE_HEIGHT}
	 * 
	 * @return The height of the console in characters or -1 if the height
	 *         cannot be determined.
	 */
	public static int getConsoleHeight() {
		int theConsoleHeight = TerminalFactory.get().getHeight();
		String theResult = System.getProperty( SystemConsts.SYS_PROP_CONSOLE_HEIGHT );
		if ( theResult != null ) {
			try {
				theConsoleHeight = Integer.valueOf( theResult.toString() );
			}
			catch ( NumberFormatException e ) {}
		}
		theResult = System.getProperty( "LINES" );
		if ( theResult != null ) {
			try {
				theConsoleHeight = Integer.valueOf( theResult.toString() );
			}
			catch ( NumberFormatException e ) {}
		}
		if ( theConsoleHeight <= 1 ) {
			theConsoleHeight = -1;
		}
		return theConsoleHeight;
	}

	/**
	 * Executes a command and returns the output.
	 * 
	 * @return Null if execution failed, else the according output. An empty
	 *         {@link String} stands fur successful execution.
	 */
	public static String exec( String aCommand ) {
		try {
			StringBuilder theBuilder = new StringBuilder();
			Runtime theRuntime = Runtime.getRuntime();
			Process theProcess = theRuntime.exec( aCommand );
			theProcess.waitFor();
			BufferedReader theReader = new BufferedReader( new InputStreamReader( theProcess.getInputStream() ) );
			String line;
			while ( (line = theReader.readLine()) != null ) {
				theBuilder.append( line );
			}
			theReader.close();
			return theBuilder.toString();
		}
		catch ( InterruptedException | IOException e ) {
			return null;
		}
	}

	/**
	 * Determines whether ANSI escape sequences are supported by the console.
	 * 
	 * @return True in case ANSI escape sequences are supported, else false.
	 */
	public static boolean isAnsiSupported() {
		if ( isCygwin() ) return true;
		Terminal theTerminal = TerminalFactory.get();
		return (theTerminal.isAnsiSupported());
	}

	/**
	 * Returns the operating saystem's temp folder.
	 * 
	 * @return The folder used by the OS for storing temporary files.
	 */
	public static File getTempDir() {
		String tempDir = System.getProperty( SystemConsts.SYS_PROP_JAVA_IO_TMPDIR );
		File theTempDir = new File( tempDir );
		return theTempDir;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Determines the encoding of the system (system's console) in use as of
	 * {@link System#out}.
	 * 
	 * @return The encoding (of the console).
	 */
	public static String getSystemEncoding() {
		OutputStreamWriter theOutWriter = new OutputStreamWriter( System.out );
		return theOutWriter.getEncoding();
	}

	/**
	 * Tries to determine whether the command line interpreter (CLI) is a Cygwin
	 * one.
	 * 
	 * @return True in case we think we are running in Cygwin. Use
	 *         {@link #getCommandLineInterpreter()} to test for the type of CLI,
	 *         in case you got to distinguish the
	 *         {@link CommandLineInterpreter#SHELL} type, then use this
	 *         {@link #isCygwin()} method.
	 */
	private static boolean isCygwin() {
		if ( getOperatingSystem() == OperatingSystem.WINDOWS ) {
			String theUname = getUname();
			return (theUname != null && theUname.toLowerCase().indexOf( "cygwin" ) != -1);
		}
		return false;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
