// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime.impls;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Properties;

import org.refcodes.collection.impls.PropertiesImpl;
import org.refcodes.data.RuntimeConsts;
import org.refcodes.runtime.RuntimeUtility;

/**
 * The {@link RuntimePropertiesImpl} load the first properties file found (with
 * "key=value" entries) in one of several default locations determined by the
 * {@link RuntimeUtility#toAppConfigDirs()} method.
 *
 * At the time of this writing (2015-11-12), the locations which to seek for the
 * first occurrence of the properties file are determined as follows:
 *
 * The applications base directory (where your JAR or your classes reside) is
 * taken and a list of directories relative to this base directory is generated
 * as defined in the {@link RuntimeConsts#CONFIG_DIR_NAMES}: The actual
 * directories being returned (in case them exist) are as follows, relative to
 * your applications base directory:
 * 
 * <ul>
 * <li>../config"</li>
 * <li>../etc"</li>
 * <li>../settings"</li>
 * <li>../.config"</li>
 * <li>../settings"</li>
 * <li>."</li>
 * <li>./config"</li>
 * <li>./etc"</li>
 * <li>./settings"</li>
 * <li>./.config"</li>
 * <li>./settings"</li>
 * 
 * In case you pass a JVM argument via "-Dconfig.dir=path_to_your_config_dir"
 * (where path_to_your_config_dir stands for the path to the directory where you
 * placed configuration files), then your path_to_your_config_dir is placed
 * first in the list (in case the directory exists).See
 * {@link RuntimeConsts#SYS_PROP_CONFIG_DIR}
 */
public class RuntimePropertiesImpl extends PropertiesImpl {

	/**
	 * Loads a properties file from the first folder containing such a file as
	 * of the specification for the method
	 * {@link RuntimeUtility#toAppConfigDirs()}.
	 * 
	 * @param aPropertiesFileName The filename of the properties file to load.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 */
	public RuntimePropertiesImpl( String aPropertiesFileName ) throws IOException {
		List<File> theConfigDirs = RuntimeUtility.toAppConfigDirs();
		File eFile;
		out: for ( File eDir : theConfigDirs ) {
			eFile = new File( eDir, aPropertiesFileName );
			if ( eFile.exists() && eFile.isFile() && eFile.canRead() ) {
				InputStream theInputStream = new BufferedInputStream( new FileInputStream( eFile ) );
				Properties theProperties = new Properties();
				theProperties.load( theInputStream );
				for ( Object eKey : theProperties.keySet() ) {
					_properties.put( (String) eKey, theProperties.getProperty( (String) eKey ) );
					break out;
				}
			}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
