// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.tabular;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.NoSuchElementException;

import org.refcodes.data.Delimiter;
import org.refcodes.data.Text;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.textual.CsvBuilderImpl;
import org.refcodes.textual.CsvEscapeMode;

/**
 * The {@link CsvInputStreamRecordsImpl} is an implementation of the
 * {@link Records} interface and provides functionality to parse CSV input
 * streams.
 * <p>
 * TODO: This class is still to be unit tested.
 *
 * @param <T> The type managed by the {@link Records}.
 */
public class CsvInputStreamRecordsImpl<T> implements InputStreamRecords<T> {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Header<T> _header = null;

	private BufferedReader _reader;

	private String _nextLine;

	private char _csvDelimiter;

	private int _line = -1;

	private boolean _isStrict;

	private long _erroneousRecordCount = 0;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link CsvInputStreamRecordsImpl} with the given parameters.
	 * Internally {@link Column} instances are generated according to the keys
	 * found in the CSV top line. The {@link Column} instances are required to
	 * convert the CSV line values. If a {@link Header} is provided, then the
	 * {@link Header} is used for generating the {@link Column} instances
	 * instead of the top line of the CSV file.
	 * 
	 * @param aHeader The {@link Header} to use when parsing the lines retrieved
	 *        from the {@link InputStream}.
	 * 
	 * @param aCsvInputStream The CSV {@link InputStream} which to parse.
	 * 
	 * @throws IOException in case there were problems working with the given
	 *         {@link InputStream}.
	 */
	public CsvInputStreamRecordsImpl( Header<T> aHeader, InputStream aCsvInputStream ) throws IOException {
		this( aHeader, null, aCsvInputStream, Delimiter.CSV.getChar(), true );
	}

	/**
	 * Constructs a {@link CsvInputStreamRecordsImpl} with the given parameters.
	 * Internally {@link Column} instances are generated according to the keys
	 * found in the CSV top line. The {@link Column} instances are required to
	 * convert the CSV line values. If a {@link Header} is provided, then the
	 * {@link Header} is used for generating the {@link Column} instances
	 * instead of the top line of the CSV file.
	 * 
	 * @param aHeader The {@link Header} to use when parsing the lines retrieved
	 *        from the {@link InputStream}.
	 * 
	 * @param aCsvInputStream The CSV {@link InputStream} which to parse.
	 * 
	 * @param isStrict When true, then parsing will abort with an exception in
	 *        case of parsing problems, else parsing is gracefully continued and
	 *        erroneous records are skipped. The error count
	 *        {@link #getErroneousRecordCount()} is incremented by each
	 *        erroneous {@link Record}.
	 * 
	 * @throws IOException in case there were problems working with the given
	 *         {@link InputStream}.
	 */
	public CsvInputStreamRecordsImpl( Header<T> aHeader, InputStream aCsvInputStream, boolean isStrict ) throws IOException {
		this( aHeader, null, aCsvInputStream, Delimiter.CSV.getChar(), isStrict );
	}

	/**
	 * Constructs a {@link CsvInputStreamRecordsImpl} with the given parameters.
	 * Internally {@link Column} instances are generated according to the keys
	 * found in the CSV top line. The {@link Column} instances are required to
	 * convert the CSV line values. If a {@link Header} is provided, then the
	 * {@link Header} is used for generating the {@link Column} instances
	 * instead of the top line of the CSV file.
	 *
	 * @param aHeader The {@link Header} to use when parsing the lines retrieved
	 *        from the {@link InputStream}.
	 * @param aCsvInputStream The CSV {@link InputStream} which to parse.
	 * @param aCsvDelimiter The delimiter being expected for the CSV input
	 *        stream.
	 * @throws IOException in case there were problems working with the given
	 *         {@link InputStream}.
	 */
	public CsvInputStreamRecordsImpl( Header<T> aHeader, InputStream aCsvInputStream, char aCsvDelimiter ) throws IOException {
		this( aHeader, null, aCsvInputStream, aCsvDelimiter, true );
	}

	/**
	 * Constructs a {@link CsvInputStreamRecordsImpl} with the given parameters.
	 * Internally {@link Column} instances are generated according to the keys
	 * found in the CSV top line. The {@link Column} instances are required to
	 * convert the CSV line values. If a {@link Header} is provided, then the
	 * {@link Header} is used for generating the {@link Column} instances
	 * instead of the top line of the CSV file.
	 * 
	 * @param aHeader The {@link Header} to use when parsing the lines retrieved
	 *        from the {@link InputStream}.
	 * 
	 * @param aCsvInputStream The CSV {@link InputStream} which to parse.
	 * 
	 * @param aCsvDelimiter The delimiter being expected for the CSV input
	 *        stream.
	 * 
	 * @param isStrict When true, then parsing will abort with an exception in
	 *        case of parsing problems, else parsing is gracefully continued and
	 *        erroneous records are skipped. The error count
	 *        {@link #getErroneousRecordCount()} is incremented by each
	 *        erroneous {@link Record}.
	 * 
	 * @throws IOException in case there were problems working with the given
	 *         {@link InputStream}.
	 */
	public CsvInputStreamRecordsImpl( Header<T> aHeader, InputStream aCsvInputStream, char aCsvDelimiter, boolean isStrict ) throws IOException {
		this( aHeader, null, aCsvInputStream, aCsvDelimiter, isStrict );
	}

	/**
	 * Constructs a {@link CsvInputStreamRecordsImpl} with the given parameters.
	 * This constructor supports a {@link ColumnFactory} for creating
	 * {@link Column} instance according to the keys found in the CSV top line.
	 * The {@link Column} instances are required to convert the CSV line values
	 * from the storage format to the actual required type.
	 * 
	 * @param aColumnFactory A {@link ColumnFactory} to be used to generate
	 *        {@link Column} instances from the top line of the CSF file,
	 *        required for parsing the CSV lines and converting them to
	 *        {@link Record} instances.
	 * 
	 * @param aCsvInputStream The CSV {@link InputStream} which to parse.
	 * 
	 * @throws IOException in case there were problems working with the given
	 *         {@link InputStream}.
	 */
	public CsvInputStreamRecordsImpl( ColumnFactory<T> aColumnFactory, InputStream aCsvInputStream ) throws IOException {
		this( null, aColumnFactory, aCsvInputStream, Delimiter.CSV.getChar(), true );
	}

	/**
	 * Constructs a {@link CsvInputStreamRecordsImpl} with the given parameters.
	 * This constructor supports a {@link ColumnFactory} for creating
	 * {@link Column} instance according to the keys found in the CSV top line.
	 * The {@link Column} instances are required to convert the CSV line values
	 * from the storage format to the actual required type.
	 * 
	 * @param aColumnFactory A {@link ColumnFactory} to be used to generate
	 *        {@link Column} instances from the top line of the CSF file,
	 *        required for parsing the CSV lines and converting them to
	 *        {@link Record} instances.
	 * 
	 * @param aCsvInputStream The CSV {@link InputStream} which to parse.
	 * 
	 * @param isStrict When true, then parsing will abort with an exception in
	 *        case of parsing problems, else parsing is gracefully continued and
	 *        erroneous records are skipped. The error count
	 *        {@link #getErroneousRecordCount()} is incremented by each
	 *        erroneous {@link Record}.
	 * 
	 * @throws IOException in case there were problems working with the given
	 *         {@link InputStream}.
	 */
	public CsvInputStreamRecordsImpl( ColumnFactory<T> aColumnFactory, InputStream aCsvInputStream, boolean isStrict ) throws IOException {
		this( null, aColumnFactory, aCsvInputStream, Delimiter.CSV.getChar(), isStrict );
	}

	/**
	 * Constructs a {@link CsvInputStreamRecordsImpl} with the given parameters.
	 * This constructor supports a {@link ColumnFactory} for creating
	 * {@link Column} instance according to the keys found in the CSV top line.
	 * The {@link Column} instances are required to convert the CSV line values
	 * from the storage format to the actual required type.
	 * 
	 * @param aColumnFactory A {@link ColumnFactory} to be used to generate
	 *        {@link Column} instances from the top line of the CSF file,
	 *        required for parsing the CSV lines and converting them to
	 *        {@link Record} instances.
	 * 
	 * @param aCsvInputStream The CSV {@link InputStream} which to parse.
	 * 
	 * @param aCsvDelimiter The delimiter being expected for the CSV input
	 *        stream.
	 * 
	 * @throws IOException in case there were problems working with the given
	 *         {@link InputStream}.
	 */
	public CsvInputStreamRecordsImpl( ColumnFactory<T> aColumnFactory, InputStream aCsvInputStream, char aCsvDelimiter ) throws IOException {
		this( null, aColumnFactory, aCsvInputStream, aCsvDelimiter, true );
	}

	/**
	 * Constructs a {@link CsvInputStreamRecordsImpl} with the given parameters.
	 * This constructor supports a {@link ColumnFactory} for creating
	 * {@link Column} instance according to the keys found in the CSV top line.
	 * The {@link Column} instances are required to convert the CSV line values
	 * from the storage format to the actual required type.
	 * 
	 * @param aColumnFactory A {@link ColumnFactory} to be used to generate
	 *        {@link Column} instances from the top line of the CSF file,
	 *        required for parsing the CSV lines and converting them to
	 *        {@link Record} instances.
	 * 
	 * @param aCsvInputStream The CSV {@link InputStream} which to parse.
	 * 
	 * @param aCsvDelimiter The delimiter being expected for the CSV input
	 *        stream.
	 * 
	 * @param isStrict When true, then parsing will abort with an exception in
	 *        case of parsing problems, else parsing is gracefully continued and
	 *        erroneous records are skipped. The error count
	 *        {@link #getErroneousRecordCount()} is incremented by each
	 *        erroneous {@link Record}.
	 * 
	 * @throws IOException in case there were problems working with the given
	 *         {@link InputStream}.
	 */
	public CsvInputStreamRecordsImpl( ColumnFactory<T> aColumnFactory, InputStream aCsvInputStream, char aCsvDelimiter, boolean isStrict ) throws IOException {
		this( null, aColumnFactory, aCsvInputStream, aCsvDelimiter, isStrict );
	}

	/**
	 * Constructs a {@link CsvInputStreamRecordsImpl} with the given parameters.
	 * This constructor supports a {@link ColumnFactory} for creating
	 * {@link Column} instance according to the keys found in the CSV top line.
	 * The {@link Column} instances are required to convert the CSV line values
	 * from the storage format to the actual required type. If a {@link Header}
	 * is provided, then the {@link Header} is used for generating the
	 * {@link Column} instances instead of the top line of the CSV file.
	 * 
	 * @param aHeader The {@link Header} to use when parsing the lines retrieved
	 *        from the {@link InputStream}.
	 * 
	 * @param aColumnFactory A {@link ColumnFactory} to be used to generate
	 *        {@link Column} instances from the top line of the CSF file,
	 *        required for parsing the CSV lines and converting them to
	 *        {@link Record} instances.
	 * 
	 * @param aCsvInputStream The CSV {@link InputStream} which to parse.
	 * 
	 * @param aCsvDelimiter The delimiter being expected for the CSV input
	 *        stream.
	 * 
	 * @param isStrict When true, then parsing will abort with an exception in
	 *        case of parsing problems, else parsing is gracefully continued and
	 *        erroneous records are skipped. The error count
	 *        {@link #getErroneousRecordCount()} is incremented by each
	 *        erroneous {@link Record}.
	 * 
	 * @throws IOException in case there were problems working with the given
	 *         {@link InputStream}.
	 */
	protected CsvInputStreamRecordsImpl( Header<T> aHeader, ColumnFactory<T> aColumnFactory, InputStream aCsvInputStream, char aCsvDelimiter, boolean isStrict ) throws IOException {
		_isStrict = isStrict;
		_reader = new BufferedReader( new InputStreamReader( aCsvInputStream ) );
		// reader = new BufferedReader( new FileReader( aCsvFile ) );
		if ( aHeader == null ) {
			String theHeaderLine = _reader.readLine();
			_line++;
			if ( theHeaderLine != null ) {
				List<String> theHeaderKeys = new CsvBuilderImpl().withCsvEscapeMode( CsvEscapeMode.ESCAPED ).withRecord( theHeaderLine ).withDelimiterChar( aCsvDelimiter ).toFields();
				_header = TabularUtility.toHeader( theHeaderKeys, aColumnFactory );
			}
		}
		else {
			_header = aHeader;
		}
		_nextLine = _reader.readLine();
		_line++;
		_csvDelimiter = aCsvDelimiter;
	}

	// /////////////////////////////////////////////////////////////////////////
	// ITERATOR:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasNext() {
		if ( _header == null ) {
			return false;
		}
		return (_nextLine != null);
	}

	/**
	 * Next.
	 *
	 * @return the record
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Record<T> next() {
		if ( _nextLine == null || _header == null ) {
			throw new NoSuchElementException( "There is no more element beyond line <" + _line + "> in this iterator." );
		}
		List<Field<T>> theItems = new ArrayList<Field<T>>();
		List<String> theValues = new CsvBuilderImpl().withCsvEscapeMode( CsvEscapeMode.ESCAPED ).withRecord( _nextLine ).withDelimiterChar( _csvDelimiter ).toFields();
		T eValue = null;
		String eString;
		List<String> eStringList;
		String eStringValue;
		String[] eStringValues;
		Column<?> eColumn;
		int theSize = _header.size() <= theValues.size() ? _header.size() : theValues.size();
		for ( int i = 0; i < theSize; i++ ) {
			eValue = null;
			eString = theValues.get( i );
			eColumn = _header.get( i );
			try {
				// Do we have an array type for the given column?
				if ( eColumn.getType().isArray() ) {
					eStringList = new CsvBuilderImpl().withCsvEscapeMode( CsvEscapeMode.ESCAPED ).withRecord( eString ).withDelimiterChar( Delimiter.ARRAY.getChar() ).toFields();
					eStringValues = eStringList.toArray( new String[eStringList.size()] );
					eValue = (T) eColumn.fromStorageStrings( eStringValues );
				}
				// We have a plain type for the given column:
				else {
					eStringValue = eString;
					eValue = (T) eColumn.fromStorageString( eStringValue );
				}
			}
			catch ( ParseException e ) {
				_erroneousRecordCount++;
				if ( _isStrict ) {
					throw new NoSuchElementException( "A parse exception at line <" + _line + "> occurred while parsing the value \"" + eString + "\" for key \"" + eColumn.getKey() + "\" (type = " + eColumn.getType().getName() + ")\", the line is \"" + _nextLine + "\": " + ExceptionUtility.toMessage( e ) );
				}
			}
			catch ( IndexOutOfBoundsException e ) {
				_erroneousRecordCount++;
				if ( _isStrict ) {
					throw new NoSuchElementException( "An index out of bounds exception at line <" + _line + "> occurred while parsing the value for key \"" + eColumn.getKey() + "\" (type = " + eColumn.getType().getName() + "), the line is \"" + _nextLine + "\": " + ExceptionUtility.toMessage( e ) );
				}
			}
			catch ( UnsupportedOperationException e ) {
				_erroneousRecordCount++;
				if ( _isStrict ) {
					throw new NoSuchElementException( "An unsupported operation exception at line <" + _line + "> occurred while parsing the value for key \"" + eColumn.getKey() + "\" (type = " + eColumn.getType().getName() + "), the line is \"" + _nextLine + "\": " + ExceptionUtility.toMessage( e ) );
				}
			}
			catch ( NumberFormatException e ) {
				_erroneousRecordCount++;
				if ( _isStrict ) {
					throw new NoSuchElementException( "A number format exception exception at line <" + _line + "> occurred while parsing the value for key \"" + eColumn.getKey() + "\" (type = " + eColumn.getType().getName() + "), the line is \"" + _nextLine + "\": " + ExceptionUtility.toMessage( e ) );
				}
			}
			catch ( Exception e ) {
				_erroneousRecordCount++;
				if ( _isStrict ) {
					throw new NoSuchElementException( "An exception of type \"" + e.getClass().getName() + "\" at line <" + _line + "> occurred while parsing the value for key \"" + eColumn.getKey() + "\" (type = " + eColumn.getType().getName() + "), the line is \"" + _nextLine + "\": " + ExceptionUtility.toMessage( e ) );
				}
			}
			theItems.add( new FieldImpl<T>( eColumn.getKey(), eValue ) );
		}
		Record<T> theRecord = new RecordImpl<T>( theItems );
		try {
			_nextLine = _reader.readLine();
			_line++;
		}
		catch ( IOException e ) {
			_nextLine = null;
		}
		if ( !hasNext() ) {
			try {
				_reader.close();
			}
			catch ( IOException e ) {}
		}
		return theRecord;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void remove() {
		throw new UnsupportedOperationException( Text.UNSUPPORTED_OPERATION.getText() );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Header<T> getHeader() {
		return _header;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	/**
	 * Gets the erroneous record count.
	 *
	 * @return the erroneous record count
	 */
	// /////////////////////////////////////////////////////////////////////////
	@Override
	public long getErroneousRecordCount() {
		return _erroneousRecordCount;
	}
}
