// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.tabular;

import java.text.ParseException;

import org.refcodes.data.Text;
import org.refcodes.exception.ExceptionUtility;

/**
 * The Class ExceptionColumnImpl.
 */
public class ExceptionColumnImpl extends AbstractColumn<Throwable> implements Cloneable {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private PrintStackTrace _printStackTrace;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates an {@link ExceptionColumnImpl} which will consider the stack
	 * trace in its string representation {@link #toStorageString(Object)}.
	 * 
	 * @param aKey The key for the {@link ExceptionColumnImpl}.
	 */
	public ExceptionColumnImpl( String aKey ) {
		this( aKey, PrintStackTrace.EXPLODED );
	}

	/**
	 * Creates an {@link ExceptionColumnImpl} which can consider or omit the
	 * stack trace in its string representation
	 * {@link #toStorageString(Object)}.
	 *
	 * @param aKey The key for the {@link ExceptionColumnImpl}.
	 * @param aPrintStackTrace the print stack trace
	 */
	public ExceptionColumnImpl( String aKey, PrintStackTrace aPrintStackTrace ) {
		super( aKey, Throwable.class );
		_printStackTrace = aPrintStackTrace;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String[] toStorageStrings( Throwable aValue ) {
		switch ( _printStackTrace ) {
		case COMPACT:
			return new String[] {
					ExceptionUtility.toMessage( aValue ) + ": " + ExceptionUtility.toStackTrace( aValue )
			};
		case NONE:
			return new String[] {
					ExceptionUtility.toMessage( aValue )
			};
		case EXPLODED:
		default:
			return new String[] {
					ExceptionUtility.toMessage( aValue ) + ":\n" + ExceptionUtility.toStackTrace( aValue )
			};
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized Throwable fromStorageStrings( String[] aStringValues ) throws ParseException {
		throw new UnsupportedOperationException( Text.UNSUPPORTED_OPERATION.getText() );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toPrintable( Throwable aValue ) {
		switch ( _printStackTrace ) {
		case COMPACT:
			return ExceptionUtility.toMessage( aValue ) + ": " + ExceptionUtility.toStackTrace( aValue );
		case NONE:
			return ExceptionUtility.toMessage( aValue );
		case EXPLODED:
		default:
			return ExceptionUtility.toMessage( aValue ) + ":\n" + ExceptionUtility.toStackTrace( aValue );
		}

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}
}