// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.time;

import java.time.DateTimeException;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.Date;

/**
 * The {@link DateFormats} represent sets of {@link DateFormat} definitions.
 * Such a set is most useful to parse back a {@link String} into a {@link Date}
 * when there are multiple possible {@link String} representations. E.g. a
 * cookie's date representation might be either as defined by the
 * {@link DateFormat#NETSCAPE_COOKIE_DATE_FORMAT} or by the
 * {@link DateFormat#ALTERNATE_COOKIE_DATE_FORMAT}.
 */
public enum DateFormats {

	// /////////////////////////////////////////////////////////////////////////
	// DATE FORMATS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Sound set of common {@link DateFormat} definitions with no special
	 * addressee.
	 */
	DEFAULT_DATE_FORMATS(new DateTimeFormatter[] {
			DateFormat.NORM_DATE_FORMAT.getFormatter(), DateFormat.MIN_DATE_FORMAT.getFormatter(), DateFormat.ALTERNATE_DATE_FORMAT.getFormatter(), DateFormat.DE_DATE_FORMAT.getFormatter(), DateFormat.NETSCAPE_COOKIE_DATE_FORMAT.getFormatter()
	}),

	/**
	 * Sound set of common short {@link DateFormat} definitions with no special
	 * addressee.
	 */
	DEFAULT_DATE_FORMATS_SHORT(new DateTimeFormatter[] {
			DateFormat.MIN_DATE_FORMAT.getFormatter(), DateFormat.NORM_DATE_FORMAT.getFormatter(), DateFormat.ALTERNATE_DATE_FORMAT.getFormatter(), DateFormat.DE_DATE_FORMAT.getFormatter()
	}),

	/**
	 * Set of {@link DateFormat} definitions commonly used by cookies.
	 */
	COOKIE_DATE_FORMATS(new DateTimeFormatter[] {
			DateFormat.NETSCAPE_COOKIE_DATE_FORMAT.getFormatter(), DateFormat.ALTERNATE_COOKIE_DATE_FORMAT.getFormatter()
	});

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private DateTimeFormatter[] _dateFormats;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// ////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new date formats.
	 *
	 * @param aDateFormats the date formats
	 */
	private DateFormats( DateTimeFormatter[] aDateFormats ) {
		_dateFormats = aDateFormats;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// ////////////////////////////////////////////////////////////////////////

	/**
	 * Gets the date formats.
	 *
	 * @return the date formats
	 */
	public DateTimeFormatter[] getDateFormats() {
		return _dateFormats;
	}

	/**
	 * Creates a date from the provided {@link String} using the date formats as
	 * retrieved by the {@link #getDateFormats()} method. If one date format
	 * fails, then the next one is used to parse the date text.
	 *
	 * @param aDateString The date text to be converted to a {@link Date}
	 *        instance.
	 * @return The {@link Date} instance as of the date text.
	 * @throws DateTimeException the date time exception
	 */
	public Date toDate( String aDateString ) throws DateTimeException {
		DateTimeException theFirstException = null;
		for ( DateTimeFormatter eDateFormat : getDateFormats() ) {
			try {

				Instant theInstant = Instant.from( eDateFormat.parse( aDateString ) );
				return new Date( theInstant.toEpochMilli() );
			}
			catch ( DateTimeException e ) {
				if ( theFirstException != null ) theFirstException = e;
			}
		}
		if ( theFirstException != null ) throw theFirstException;
		throw new DateTimeException( "Unable to parse date for date string <" + aDateString + ">." );
	}
}
