// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.time;

import java.text.ParseException;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.Date;

/**
 * The Class DateUtility.
 */
public final class DateUtility {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Private empty constructor to prevent instantiation as of being a utility
	 * with just static public methods.
	 */
	private DateUtility() {}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Parses a {@link String} to retrieve a date from it, using the provided
	 * {@link DateTimeFormatter} instances. The first one being able to parse a
	 * {@link Date} from the {@link String} will be used and that {@link Date}
	 * is returned. If none {@link DateTimeFormatter} instances were able to
	 * parse the provided {@link String}, then the first (as being provided in
	 * the array) {@link DateTimeFormatter}'s parse exception is thrown.
	 * 
	 * @param aValue The date {@link String} to be parsed.
	 * 
	 * @param aDateTimeFormatters The {@link DateTimeFormatter} instances to be
	 *        tried out, starting with the first one.
	 * 
	 * @return A {@link Date} retrieved from the given {@link String}.
	 * 
	 * @throws ParseException Thrown in case none of the provided
	 *         {@link DateTimeFormatter} instances was able to parse the string.
	 *         Then the first caught {@link ParseException} is thrown.
	 */
	public static Date toDate( String aValue, DateTimeFormatter[] aDateTimeFormatters ) throws ParseException {
		DateTimeException theParseException = null;
		Instant eInstant;
		for ( int i = 0; i < aDateTimeFormatters.length; i++ ) {
			try {
				eInstant = Instant.from( aDateTimeFormatters[i].parse( aValue ) );
				return new Date( eInstant.toEpochMilli() );
			}
			catch ( DateTimeException e ) {
				if ( theParseException == null ) {
					theParseException = e;
				}
			}
		}
		throw theParseException;
	}
}
