/*
 * TransportCursor.java February 2007
 *
 * Copyright (C) 2001, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.io.IOException;
import java.nio.ByteBuffer;

import org.simpleframework.transport.Transport;

/**
 * The <code>TransportCursor</code> object represents a cursor that
 * can read and buffer data from an underlying transport. If the
 * number of bytes read from the cursor is more than required for
 * the HTTP request then those bytes can be pushed back in to the
 * cursor using the <code>reset</code> method. This will only allow
 * the last read to be reset within the cursor safely. 
 * 
 * @author Niall Gallagher
 * 
 * @see org.simpleframework.transport.Transport
 */
class TransportCursor implements Cursor {

   /**
    * This is the underlying transport to read the bytes from. 
    */
   private Transport transport;

   /**
    * This is used to store the bytes read from the transport.
    */
   private ByteBuffer buffer;

   /**
    * This is used to determine if the transport has been closed.
    */ 
   private boolean closed;
   
   /**
    * This is used to determine if a read should be performed.
    */
   private boolean reject;

   /**
    * This represents the number of bytes that are ready to read.
    */
   private int count;
   
   /**
    * Constructor for the <code>TransportCursor</code> object. This
    * requires a transport to read the bytes from. By default this 
    * will create a buffer of two kilobytes to read the input in to
    * which ensures several requests can be read at once.
    * 
    * @param transport this is the underlying transport to use
    * @param lease this is the lease to keep the transport open
    */
   public TransportCursor(Transport transport) {
      this(transport, 2048);
   }

   /**
    * Constructor for the <code>TransportCursor</code> object. This
    * requires a transport to read the bytes from. By default this 
    * will create a buffer of of the specified size to read the 
    * input in to which enabled bytes to be buffered internally.
    * 
    * @param transport this is the underlying transport to use
    * @param size this is the size of the internal buffer to use
    */   
   public TransportCursor(Transport transport, int size) {
      this.buffer = ByteBuffer.allocate(size);
      this.transport = transport;
   }
   
   /**
    * Determines whether the cursor is still open. The cursor is
    * considered open if there are still bytes to read. If there is
    * still bytes buffered and the underlying transport is closed
    * then the cursor is still considered open. 
    * 
    * @return true if the read method does not return a -1 value
    */   
   public boolean isOpen() throws IOException {
      return count != -1;
   }

   /**
    * Determines whether the cursor is ready for reading. When the
    * cursor is ready then it guarantees that some amount of bytes
    * can be read from the underlying stream without blocking.
    *
    * @return true if some data can be read without blocking
    */    
   public boolean isReady() throws IOException {
      return ready() > 0;
   }

   /**
    * Reads a block of bytes from the underlying stream. This will
    * read up to the requested number of bytes from the underlying
    * stream. If there are no ready bytes on the stream this will
    * block, much like the <code>InputStream</code> a minus one is
    * returned if there are no more bytes to be read.
    *
    * @param data this is the array to read the bytes in to 
    *
    * @return this returns the number of bytes read from the stream 
    */    
   public int read(byte[] array) throws IOException {
      return read(array, 0, array.length);
   }

   /**
    * Reads a block of bytes from the underlying stream. This will
    * read up to the requested number of bytes from the underlying
    * stream. If there are no ready bytes on the stream this will
    * block, much like the <code>InputStream</code> a minus one is
    * returned if there are no more bytes to be read.
    *
    * @param data this is the array to read the bytes in to
    * @param off this is the offset to begin writing the bytes to
    * @param len this is the number of bytes that are requested 
    *
    * @return this returns the number of bytes read from the stream 
    */    
   public int read(byte[] array, int off, int len) throws IOException {
      if(count <= 0) { // has the channel ended
         return count;
      }
      int size = Math.min(len, count); // get the minimum

      if(size > 0) {
         buffer.get(array, off, size); // get the bytes
         count -= size;
      }
      return Math.max(0, size);
   }

   /**
    * Provides the number of bytes that can be read from the stream
    * without blocking. This is typically the number of buffered or
    * available bytes within the stream. When this reaches zero then
    * the cursor may perform a blocking read.
    *
    * @return the number of bytes that can be read without blocking
    */    
   public int ready() throws IOException {
      if(count < 0) {
         return count;
      }
      if(reject) {
         reject = false; // clear rejected flag
         return peek();
      }
      if(count > 0) { // if the are ready bytes don't read
         return count;      
      }
      return peek();
   }

   /**
    * Provides the number of bytes that can be read from the stream
    * without blocking. This is typically the number of buffered or
    * available bytes within the stream. When this reaches zero then
    * the cursor may perform a blocking read.
    *
    * @return the number of bytes that can be read without blocking
    */  
   private int peek() throws IOException {
      if(count <= 0) { // reset the buffer for filling
         buffer.clear();
      }
      if(count > 0) {
         buffer.compact(); // compact the buffer
      }
      count += transport.read(buffer); // how many were read

      if(count > 0) {
         buffer.flip(); // if there is something then flip
      }
      if(count < 0) { // close when stream is fully read
         close();
      }
      return count;
   }

   /**
    * Moves the cursor backward within the stream. This ensures 
    * that any bytes read from the last read can be pushed back
    * in to the stream so that they can be read again. This will
    * throw an exception if the reset can not be performed.
    *
    * @param len this is the number of bytes to reset back
    *
    * @return this is the number of bytes that have been reset
    */
   public int reset(int size) throws IOException {
      int mark = buffer.position();

      if(count == 0) {
         reject = true; // force a new read
      }
      if(mark > 0) {
         buffer.position(mark - size);
         count += size;
      }
      return size;
   }

   /**
    * This is used to close the underlying transport. This is used
    * when the transport returns a negative value, indicating that
    * the client has closed the connection on the other side. If
    * this is invoked the read method returns -1 and the cursor
    * is not longer open, further bytes can no longer be read.
    */
   public void close() throws IOException {
      if(!closed) {
        transport.close();
        closed = true;
        count = -1;
      }
   }
}


