/*
 * ProcessorServer.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.transport;

import java.io.IOException;

import org.simpleframework.transport.reactor.Operation;

/**
 * The <code>ProcessorServer</code> is used to convert pipelines
 * to transports. This simply acts as an adapter to a processor
 * which converts a connected pipeline to a <code>Transport</code>
 * which can be used to read and write data. Conversion of the
 * pipeline is performed only once per connection.
 * 
 * @author Niall Gallagher
 */
public class ProcessorServer implements Server {

   /**
    * This is the factory used to create the required operations.
    */
   private final OperationFactory factory;
   
   /**
    * This is the transport used to process complete transports.
    */
   private final Processor processor;
   
   /**
    * This is the processor used to process transport objects.
    */
   private final Negotiator negotiator;

   /**
    * Constructor for the <code>ProcessorServer</code> object. 
    * The transport processor is used to process plain connections
    * and wrap those connections in a <code>Transport</code> that
    * can be used to send and receive data to and from.
    * 
    * @param processor this is used to process transports
    */
   public ProcessorServer(Processor processor) throws IOException {
      this(processor, 8);
   }
   
   /**
    * Constructor for the <code>ProcessorServer</code> object. 
    * The transport processor is used to process plain connections
    * and wrap those connections in a <code>Transport</code> that
    * can be used to send and receive data to and from.
    * 
    * @param processor this is used to process transports
    * @param count this is the number of threads this will use
    */
   public ProcessorServer(Processor processor, int count) throws IOException {
      this(processor, count, 20480);
   }
      
   /**
    * Constructor for the <code>ProcessorServer</code> object. 
    * The transport processor is used to process plain connections
    * and wrap those connections in a <code>Transport</code> that
    * can be used to send and receive data to and from.
    * 
    * @param processor this is used to process transports
    * @param count this is the number of threads this will use
    * @param limit this is the threshold for asynchronous buffers
    */
   public ProcessorServer(Processor processor, int count, int limit) throws IOException {
      this.negotiator = new SecureNegotiator(processor, count);
      this.factory = new OperationFactory(negotiator, limit);
      this.processor = processor;
   }

   /**
    * Used to process the <code>Pipeline</code> which is a full duplex 
    * communication link that may contain several HTTP requests. This 
    * will be used to read the requests from the <code>Pipeline</code> 
    * and to pass these requests to a <code>Container</code> for 
    * processing.
    * <p>
    * Typical usage of this method is to accept multiple pipeline 
    * objects, each representing a unique HTTP channel to the client,
    * and process requests from those pipelines concurrently.  
    *
    * @param socket this is the connected HTTP pipeline to process
    */    
   public void process(Socket socket) throws IOException {
      Operation task = factory.getInstance(socket);
      
      if(task != null) {
         negotiator.process(task);
      }
   }
   
   /**
    * This method is used to stop the <code>Processor</code> such that
    * it will accept no more pipelines. Stopping the processor ensures
    * that all resources occupied will be released. This is required
    * so that all threads are stopped, and all memory is released.
    * <p>
    * Typically this method is called once all connections to the
    * server have been stopped. As a final act of shutting down the
    * entire server all threads must be stopped, this allows collection
    * of unused memory and the closing of file and socket resources.
    */    
   public void stop() throws IOException {
      processor.stop();
      negotiator.stop();
   }
 }
