/*
 * LeaseMap.java May 2004
 *
 * Copyright (C) 2004, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.util.lease;

import java.util.concurrent.ConcurrentHashMap;

/**
 * The <code>LeaseMap</code> object is used to map lease keys to the
 * lease objects managing those objects. This allows components that
 * are using the leasing framework to associate an object with its
 * lease and vice versa. Such a capability enables lease renewals to
 * be performed without the need for a direct handle on the lease.
 * 
 * @author Niall Gallagher
 */
public class LeaseMap<T> extends ConcurrentHashMap<T, Lease<T>> {
   
   /**
    * Constructor for the <code>LeaseMap</code> object. This will
    * create a map for mapping leased resource keys to the leases
    * that manage them. Having such a map allows leases to be 
    * maintained without having a direct handle on the lease.
    */
   public LeaseMap() {
      super();
   }
   
   /**
    * Constructor for the <code>LeaseMap</code> object. This will
    * create a map for mapping leased resource keys to the leases
    * that manage them. Having such a map allows leases to be 
    * maintained without having a direct handle on the lease.
    * 
    * @param capacity this is the initial capacity of the map
    */
   public LeaseMap(int capacity) {
      super(capacity);
   }
   
   /**
    * This is used to acquire the <code>Lease</code> object that is
    * mapped to the specified key. Overriding this method ensures
    * that even without generic parameters a type safe method for
    * acquiring the registered lease objects can be used.
    * 
    * @param key this is the key used to acquire the lease object
    * 
    * @return this is the lease that is associated with the key
    */
   public Lease<T> get(Object key) {
      return super.get(key);
   }
   
   /**
    * This is used to remove the <code>Lease</code> object that is
    * mapped to the specified key. Overriding this method ensures
    * that even without generic parameters a type safe method for
    * removing the registered lease objects can be used.
    * 
    * @param key this is the key used to remove the lease object
    * 
    * @return this is the lease that is associated with the key
    */
   public Lease<T> remove(Object key) {
      return super.remove(key);
   }
}
