/*
 * HeaderConsumer.java February 2007
 *
 * Copyright (C) 2001, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.util.List;

import org.simpleframework.http.Cookie;

/**
 * The <code>HeaderConsumer</code> object is used to consume a HTTP
 * header from the cursor. This extends the segment consumer with
 * methods specific to the header. Also this enables session cookies
 * to be created using the cookies extracted from the header.
 * 
 * @author Niall Gallagher
 */
abstract class HeaderConsumer extends SegmentConsumer implements Header {
   
   /**
    * This is the policy object used to create the session cookie.
    */
   private Policy policy;
   
   /**
    * Constructor for the <code>HeaderConsumer</code> object. This 
    * is used to create a consumer capable of reading a header from
    * a provided cursor. All methods of the <code>Header</coder> 
    * interface are implemented in this object.
    */
   protected HeaderConsumer() {
      this.policy = new SecurePolicy(this);      
   }

   /**
    * This is used to see if there is a HTTP message header with the
    * given name in this container. If there is a HTTP message header
    * with the specified name then this returns true otherwise false.
    *
    * @param name the HTTP message header to get the value from
    *
    * @return this returns true if the HTTP message header exists
    */    
   public boolean contains(String name) {
      return header.contains(name);
   }

   /**
    * This can be used to get the date of the first message header
    * that has the specified name. This is a convenience method that 
    * avoids having to deal with parsing the value of the requested
    * HTTP message header. This returns -1 if theres no HTTP header
    * value for the specified name.
    *
    * @param name the HTTP message header to get the value from
    *
    * @return this returns the date as a long from the header value 
    */     
   public long getDate(String name) {
      return header.getDate(name);
   }

   /**
    * This can be used to get the integer of the first message header
    * that has the specified name. This is a convenience method that 
    * avoids having to deal with parsing the value of the requested
    * HTTP message header. This returns -1 if theres no HTTP header
    * value for the specified name.
    *
    * @param name the HTTP message header to get the value from
    *
    * @return this returns the date as a long from the header value 
    */    
   public int getInteger(String name) {
      return header.getInteger(name);
   }

   /**
    * This method is used to get a <code>List</code> of the names
    * for the headers. This will provide the original names for the
    * HTTP headers for the message. Modifications to the provided
    * list will not affect the header, the list is a simple copy.
    *
    * @return this returns a list of the names within the header
    */   
   public List<String> getNames() {
      return header.getNames();
   }
   
   /**
    * This is used to acquire a cookie using the name of that cookie.
    * If the cookie exists within the HTTP header then it is returned
    * as a <code>Cookie</code> object. Otherwise this method will
    * return null. Each cookie object will contain the name, value
    * and path of the cookie as well as the optional domain part.
    *
    * @param name this is the name of the cookie object to acquire
    * 
    * @return this returns a cookie object from the header or null
    */    
   public Cookie getCookie(String name) {
      return header.getCookie(name);
   }
   
   /**
    * This is used to acquire the session cookie for the request. The
    * session cookie is either sent with the HTTP request header or
    * it can be created if required. This ensures that if no cookie
    * has been sent one can be created on demand. 
    * 
    * @param create if true the session cookie will be created
    * 
    * @return the cookie associated with the session or null
    */   
   public Cookie getSession(boolean create) {
      return policy.getSession(create);
   }
   
   /**
    * This is used to acquire all cookies that were sent in the header.    
    * If any cookies exists within the HTTP header they are returned
    * as <code>Cookie</code> objects. Otherwise this method will an
    * empty list. Each cookie object will contain the name, value and 
    * path of the cookie as well as the optional domain part.
    * 
    * @return this returns all cookie objects from the HTTP header
    */    
   public List<Cookie> getCookies() {
      return header.getCookies();
   }
}
