/*
 * Negotiation.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.transport;

import java.io.IOException;

import org.simpleframework.transport.reactor.Operation;

/**
 * The <code>Negotiation</code> interface is used to represent an
 * SSL negotiation. When an operation can not be completed this
 * will allow a task to perform asynchronous operations and resume
 * the negotiation when those operations can be fulfilled.
 * 
 * @author Niall Gallagher
 */
interface Negotiation extends Operation {
   
   /**
    * This is used to resume the negotiation when an operation
    * has completed. This will continue the decrypt and encrypt
    * sequence of messages required to fulfill the negotiation.
    */
   public void resume() throws IOException;

   /**
    * This method is invoked when the negotiation is done and
    * the next phase of the connection is to take place. This
    * will typically be invoked when an SSL handshake or
    * termination exchange has completed successfully. 
    */
   public void commit() throws IOException;     
   
   /**
    * This is used to send any messages the negotiation may have.
    * If the negotiation can not send the information during its
    * execution then this method will be executed when a select
    * operation is signaled.
    * 
    * @return this returns true when the message has been sent
    */
   public boolean send() throws IOException;   
   
   /**
    * This is used to receive data from the client. If at any
    * point during the negotiation a message is required that
    * can not be read immediately this is used to asynchronously
    * read the data when a select operation is signaled.
    *  
    * @return this returns true when the message has been read
    */
   public boolean receive() throws IOException; 
}
