/*
 * Decompiled with CFR 0.152.
 */
package org.sonarsource.scanner.lib.internal.http;

import java.net.InetSocketAddress;
import java.net.Proxy;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.time.format.DateTimeParseException;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.CheckForNull;
import javax.annotation.Nullable;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonarsource.scanner.lib.internal.http.ssl.CertificateStore;
import org.sonarsource.scanner.lib.internal.http.ssl.SslConfig;
import org.sonarsource.scanner.lib.internal.util.System2;

public class HttpConfig {
    private static final Logger LOG = LoggerFactory.getLogger(HttpConfig.class);
    static final Duration DEFAULT_CONNECT_TIMEOUT = Duration.ofSeconds(5L);
    static final Duration DEFAULT_RESPONSE_TIMEOUT = Duration.ZERO;
    public static final String READ_TIMEOUT_SEC_PROPERTY = "sonar.ws.timeout";
    static final Duration DEFAULT_READ_TIMEOUT_SEC = Duration.ofSeconds(60L);
    static final int DEFAULT_PROXY_PORT = 80;
    private final String webApiBaseUrl;
    private final String restApiBaseUrl;
    @Nullable
    private final String token;
    @Nullable
    private final String login;
    private final System2 system;
    @Nullable
    private final String password;
    private final SslConfig sslConfig;
    private final Duration socketTimeout;
    private final Duration connectTimeout;
    private final Duration responseTimeout;
    @Nullable
    private final Proxy proxy;
    @Nullable
    private final String proxyUser;
    @Nullable
    private final String proxyPassword;
    private final String userAgent;
    private final boolean skipSystemTrustMaterial;

    public HttpConfig(Map<String, String> bootstrapProperties, Path sonarUserHome, System2 system) {
        this.webApiBaseUrl = StringUtils.removeEnd((String)bootstrapProperties.get("sonar.host.url"), (String)"/");
        this.restApiBaseUrl = StringUtils.removeEnd((String)bootstrapProperties.get("sonar.scanner.apiBaseUrl"), (String)"/");
        this.token = bootstrapProperties.get("sonar.token");
        this.login = bootstrapProperties.get("sonar.login");
        this.system = system;
        if (Objects.nonNull(this.login) && Objects.nonNull(this.token)) {
            LOG.warn("Both '{}' and '{}' (or the '{}' env variable) are set, but only the latter will be used.", new Object[]{"sonar.login", "sonar.token", "SONAR_TOKEN"});
        }
        this.password = bootstrapProperties.get("sonar.password");
        this.userAgent = String.format("%s/%s", bootstrapProperties.get("sonar.scanner.app"), bootstrapProperties.get("sonar.scanner.appVersion"));
        this.socketTimeout = HttpConfig.loadDuration(bootstrapProperties, "sonar.scanner.socketTimeout", READ_TIMEOUT_SEC_PROPERTY, DEFAULT_READ_TIMEOUT_SEC);
        this.connectTimeout = HttpConfig.loadDuration(bootstrapProperties, "sonar.scanner.connectTimeout", null, DEFAULT_CONNECT_TIMEOUT);
        this.responseTimeout = HttpConfig.loadDuration(bootstrapProperties, "sonar.scanner.responseTimeout", null, DEFAULT_RESPONSE_TIMEOUT);
        this.sslConfig = this.loadSslConfig(bootstrapProperties, sonarUserHome);
        this.proxy = HttpConfig.loadProxy(bootstrapProperties);
        this.proxyUser = this.loadProxyUser(bootstrapProperties);
        this.proxyPassword = this.loadProxyPassword(bootstrapProperties);
        this.skipSystemTrustMaterial = Boolean.parseBoolean((String)StringUtils.defaultIfBlank((CharSequence)bootstrapProperties.get("sonar.scanner.skipSystemTruststore"), (CharSequence)"false"));
    }

    @CheckForNull
    private String loadProxyPassword(Map<String, String> bootstrapProperties) {
        String scannerProxyPwd = bootstrapProperties.get("sonar.scanner.proxyPassword");
        return scannerProxyPwd != null ? scannerProxyPwd : this.system.getProperty("http.proxyPassword");
    }

    @CheckForNull
    private String loadProxyUser(Map<String, String> bootstrapProperties) {
        String scannerProxyUser = bootstrapProperties.get("sonar.scanner.proxyUser");
        return scannerProxyUser != null ? scannerProxyUser : this.system.getProperty("http.proxyUser");
    }

    private static Duration loadDuration(Map<String, String> bootstrapProperties, String propKey, @Nullable String deprecatedPropKey, Duration defaultValue) {
        if (bootstrapProperties.containsKey(propKey)) {
            return HttpConfig.parseDurationProperty(bootstrapProperties.get(propKey), propKey);
        }
        if (deprecatedPropKey != null && bootstrapProperties.containsKey(deprecatedPropKey)) {
            LOG.warn("Property {} is deprecated and will be removed in a future version. Please use {} instead.", (Object)deprecatedPropKey, (Object)propKey);
            return HttpConfig.parseDurationProperty(bootstrapProperties.get(deprecatedPropKey), deprecatedPropKey);
        }
        return defaultValue;
    }

    @Nullable
    private static Proxy loadProxy(Map<String, String> bootstrapProperties) {
        String proxyHost = (String)StringUtils.defaultIfBlank((CharSequence)bootstrapProperties.get("sonar.scanner.proxyHost"), null);
        if (proxyHost != null) {
            int proxyPort = bootstrapProperties.containsKey("sonar.scanner.proxyPort") ? HttpConfig.parseIntProperty(bootstrapProperties.get("sonar.scanner.proxyPort"), "sonar.scanner.proxyPort") : 80;
            return new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxyHost, proxyPort));
        }
        return null;
    }

    private static Duration parseDurationProperty(String propValue, String propKey) {
        try {
            return Duration.parse(propValue);
        }
        catch (DateTimeParseException e) {
            return Duration.ofSeconds(HttpConfig.parseIntProperty(propValue, propKey));
        }
    }

    private static int parseIntProperty(String propValue, String propKey) {
        try {
            return Integer.parseInt(propValue);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException(propKey + " is not a valid integer: " + propValue, e);
        }
    }

    private SslConfig loadSslConfig(Map<String, String> bootstrapProperties, Path sonarUserHome) {
        boolean skipJvmSslConfig = Boolean.parseBoolean((String)StringUtils.defaultIfBlank((CharSequence)bootstrapProperties.get("sonar.scanner.skipJvmSslConfig"), (CharSequence)"false"));
        CertificateStore keyStore = this.loadKeyStoreConfig(bootstrapProperties, sonarUserHome, skipJvmSslConfig);
        CertificateStore trustStore = this.loadTrustStoreConfig(bootstrapProperties, sonarUserHome, skipJvmSslConfig);
        return new SslConfig(keyStore, trustStore);
    }

    @Nullable
    private CertificateStore loadTrustStoreConfig(Map<String, String> bootstrapProperties, Path sonarUserHome, boolean skipJvmSslConfig) {
        Path trustStorePath = HttpConfig.parseFileProperty(bootstrapProperties, "sonar.scanner.truststorePath", "truststore", sonarUserHome.resolve("ssl/truststore.p12"));
        if (trustStorePath != null) {
            LOG.debug("Using scanner truststore: {}", (Object)trustStorePath);
            return new CertificateStore(trustStorePath, bootstrapProperties.get("sonar.scanner.truststorePassword"), false);
        }
        if (!skipJvmSslConfig) {
            String jvmTrustStoreProp = this.system.getProperty("javax.net.ssl.trustStore");
            if (StringUtils.isNotBlank((CharSequence)jvmTrustStoreProp)) {
                LOG.debug("Using JVM truststore: {}", (Object)jvmTrustStoreProp);
                return new CertificateStore(Paths.get(jvmTrustStoreProp, new String[0]), this.system.getProperty("javax.net.ssl.trustStorePassword"), true);
            }
            Path defaultJvmTrustStoreLocation = Paths.get(Objects.requireNonNull(this.system.getProperty("java.home")), "lib", "security", "cacerts");
            if (Files.isRegularFile(defaultJvmTrustStoreLocation, new LinkOption[0])) {
                LOG.debug("Using JVM default truststore: {}", (Object)defaultJvmTrustStoreLocation);
                return new CertificateStore(defaultJvmTrustStoreLocation, Optional.ofNullable(this.system.getProperty("javax.net.ssl.trustStorePassword")).orElse("changeit"), true);
            }
        }
        return null;
    }

    @Nullable
    private CertificateStore loadKeyStoreConfig(Map<String, String> bootstrapProperties, Path sonarUserHome, boolean skipJvmSslConfig) {
        String jvmKeystoreProp;
        Path keyStorePath = HttpConfig.parseFileProperty(bootstrapProperties, "sonar.scanner.keystorePath", "keystore", sonarUserHome.resolve("ssl/keystore.p12"));
        if (keyStorePath != null) {
            LOG.debug("Using scanner keystore: {}", (Object)keyStorePath);
            return new CertificateStore(keyStorePath, bootstrapProperties.get("sonar.scanner.keystorePassword"), false);
        }
        if (!skipJvmSslConfig && StringUtils.isNotBlank((CharSequence)(jvmKeystoreProp = this.system.getProperty("javax.net.ssl.keyStore")))) {
            LOG.debug("Using JVM keystore: {}", (Object)jvmKeystoreProp);
            return new CertificateStore(Paths.get(jvmKeystoreProp, new String[0]), this.system.getProperty("javax.net.ssl.keyStorePassword"), true);
        }
        return null;
    }

    @Nullable
    private static Path parseFileProperty(Map<String, String> bootstrapProperties, String propKey, String labelForLogs, Path defaultPath) {
        if (bootstrapProperties.containsKey(propKey)) {
            Path keyStorePath = Paths.get(bootstrapProperties.get(propKey), new String[0]);
            if (!Files.exists(keyStorePath, new LinkOption[0])) {
                throw new IllegalArgumentException("The " + labelForLogs + " file does not exist: " + String.valueOf(keyStorePath));
            }
            return keyStorePath;
        }
        if (Files.isRegularFile(defaultPath, new LinkOption[0])) {
            return defaultPath;
        }
        return null;
    }

    public String getWebApiBaseUrl() {
        return this.webApiBaseUrl;
    }

    public String getRestApiBaseUrl() {
        return this.restApiBaseUrl;
    }

    @Nullable
    public String getToken() {
        return this.token;
    }

    @Nullable
    public String getLogin() {
        return this.login;
    }

    @Nullable
    public String getPassword() {
        return this.password;
    }

    public String getUserAgent() {
        return this.userAgent;
    }

    public SslConfig getSslConfig() {
        return this.sslConfig;
    }

    public Duration getConnectTimeout() {
        return this.connectTimeout;
    }

    public Duration getResponseTimeout() {
        return this.responseTimeout;
    }

    public Duration getSocketTimeout() {
        return this.socketTimeout;
    }

    @Nullable
    public Proxy getProxy() {
        return this.proxy;
    }

    @CheckForNull
    public String getProxyUser() {
        return this.proxyUser;
    }

    @CheckForNull
    public String getProxyPassword() {
        return this.proxyPassword;
    }

    public boolean skipSystemTruststore() {
        return this.skipSystemTrustMaterial;
    }
}

