/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.cloud.sleuth.instrument.jdbc;

import java.net.URI;
import java.sql.Connection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.sql.CommonDataSource;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.cloud.sleuth.Span;
import org.springframework.cloud.sleuth.SpanAndScope;
import org.springframework.cloud.sleuth.Tracer;
import org.springframework.cloud.sleuth.docs.AssertingSpan;
import org.springframework.cloud.sleuth.docs.AssertingSpanBuilder;
import org.springframework.cloud.sleuth.docs.DocumentedSpan;
import org.springframework.cloud.sleuth.docs.EventValue;
import org.springframework.cloud.sleuth.docs.TagKey;
import org.springframework.cloud.sleuth.instrument.jdbc.SleuthJdbcSpan;
import org.springframework.cloud.sleuth.instrument.jdbc.SpanNameProvider;
import org.springframework.cloud.sleuth.instrument.jdbc.TraceListenerStrategySpanCustomizer;
import org.springframework.cloud.sleuth.instrument.jdbc.TraceType;
import org.springframework.lang.Nullable;
import org.springframework.util.StringUtils;

class TraceListenerStrategy<CON, STMT, RS> {
    private static final Log log = LogFactory.getLog(TraceListenerStrategy.class);
    private static final Pattern URL_SERVICE_NAME_FINDER = Pattern.compile("sleuthServiceName=(.*?)(?:&|$)");
    private static final SpanNameProvider SPAN_NAME_PROVIDER = new SpanNameProvider();
    private final Map<CON, ConnectionInfo> openConnections = new ConcurrentHashMap<CON, ConnectionInfo>();
    private final ThreadLocal<ConnectionInfo> currentConnection = new ThreadLocal();
    private final List<TraceType> traceTypes;
    private final List<TraceListenerStrategySpanCustomizer<? super CommonDataSource>> customizers;
    private BeanFactory beanFactory;
    private Tracer tracer;

    TraceListenerStrategy(Tracer tracer, List<TraceType> traceTypes, List<TraceListenerStrategySpanCustomizer<? super CommonDataSource>> customizers) {
        this.traceTypes = traceTypes;
        this.customizers = customizers;
        this.tracer = tracer;
    }

    TraceListenerStrategy(BeanFactory beanFactory, List<TraceType> traceTypes, List<TraceListenerStrategySpanCustomizer<? super CommonDataSource>> customizers) {
        this.traceTypes = traceTypes;
        this.customizers = customizers;
        this.beanFactory = beanFactory;
    }

    void beforeGetConnection(CON connectionKey, @Nullable CommonDataSource dataSource, String dataSourceName) {
        if (log.isTraceEnabled()) {
            log.trace((Object)("Before get connection key [" + connectionKey + "] - current span is [" + this.getTracer().currentSpan() + "]"));
        }
        SpanAndScope spanAndScope = null;
        if (this.traceTypes.contains((Object)TraceType.CONNECTION)) {
            AssertingSpanBuilder connectionSpanBuilder = AssertingSpanBuilder.of((DocumentedSpan)SleuthJdbcSpan.JDBC_CONNECTION_SPAN, (Span.Builder)this.getTracer().spanBuilder()).name(SleuthJdbcSpan.JDBC_CONNECTION_SPAN.getName());
            connectionSpanBuilder.remoteServiceName(dataSourceName);
            connectionSpanBuilder.kind(Span.Kind.CLIENT);
            this.customizers.stream().filter(customizer -> customizer.isApplicable(dataSource)).forEach(customizer -> customizer.customizeConnectionSpan(dataSource, (Span.Builder)connectionSpanBuilder));
            AssertingSpan connectionSpan = connectionSpanBuilder.start();
            Tracer.SpanInScope scope = this.isCurrent(null) ? this.getTracer().withSpan((Span)connectionSpan) : null;
            spanAndScope = new SpanAndScope((Span)connectionSpan, scope);
            if (log.isTraceEnabled()) {
                log.trace((Object)("Started client span before connection [" + connectionSpan + "] - current span is [" + this.getTracer().currentSpan() + "]"));
            }
        }
        ConnectionInfo connectionInfo = new ConnectionInfo(spanAndScope);
        connectionInfo.remoteServiceName = dataSourceName;
        this.openConnections.put(connectionKey, connectionInfo);
        if (this.isCurrent(null)) {
            this.currentConnection.set(connectionInfo);
        }
    }

    void afterGetConnection(CON connectionKey, @Nullable Connection connection, String dataSourceName, @Nullable Throwable t) {
        if (log.isTraceEnabled()) {
            log.trace((Object)("After get connection [" + connectionKey + "]. Current span is [" + this.getTracer().currentSpan() + "]"));
        }
        ConnectionInfo connectionInfo = this.openConnections.get(connectionKey);
        SpanAndScope connectionSpan = connectionInfo.span;
        if (connection != null) {
            this.parseAndSetServerIpAndPort(connectionInfo, connection, dataSourceName);
            if (connectionSpan != null) {
                connectionSpan.getSpan().remoteServiceName(connectionInfo.remoteServiceName);
                connectionSpan.getSpan().remoteIpAndPort(connectionInfo.url.getHost(), connectionInfo.url.getPort());
            }
        } else if (t != null) {
            this.openConnections.remove(connectionKey);
            if (this.isCurrent(connectionInfo)) {
                this.currentConnection.set(null);
            }
            if (connectionSpan != null) {
                if (log.isTraceEnabled()) {
                    log.trace((Object)("Closing client span due to exception [" + connectionSpan.getSpan() + "] - current span is [" + this.getTracer().currentSpan() + "]"));
                }
                connectionSpan.getSpan().error(t);
                connectionSpan.close();
                if (log.isTraceEnabled()) {
                    log.trace((Object)("Current span [" + this.getTracer().currentSpan() + "]"));
                }
            }
        }
    }

    private boolean isCurrent(@Nullable ConnectionInfo connectionInfo) {
        return this.currentConnection.get() == connectionInfo;
    }

    void beforeQuery(CON connectionKey, STMT statementKey) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)("Before query - connection [" + connectionKey + "] and current span [" + this.getTracer().currentSpan() + "]"));
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            if (log.isTraceEnabled()) {
                log.trace((Object)"Connection may be closed after statement preparation, but before statement execution");
            }
            return;
        }
        SpanAndScope spanAndScope = null;
        if (this.traceTypes.contains((Object)TraceType.QUERY)) {
            AssertingSpanBuilder statementSpanBuilder = AssertingSpanBuilder.of((DocumentedSpan)SleuthJdbcSpan.JDBC_QUERY_SPAN, (Span.Builder)this.getTracer().spanBuilder()).name(String.format(SleuthJdbcSpan.JDBC_QUERY_SPAN.getName(), "query"));
            statementSpanBuilder.remoteServiceName(connectionInfo.remoteServiceName);
            if (connectionInfo.url != null) {
                statementSpanBuilder.remoteIpAndPort(connectionInfo.url.getHost(), connectionInfo.url.getPort());
            }
            statementSpanBuilder.kind(Span.Kind.CLIENT);
            Span statementSpan = statementSpanBuilder.start();
            Tracer.SpanInScope scope = this.isCurrent(connectionInfo) ? this.getTracer().withSpan(statementSpan) : null;
            spanAndScope = new SpanAndScope(statementSpan, scope);
            if (log.isTraceEnabled()) {
                log.trace((Object)("Started client span before query [" + statementSpan + "] - current span is [" + this.getTracer().currentSpan() + "]"));
            }
        }
        StatementInfo statementInfo = new StatementInfo(spanAndScope);
        connectionInfo.nestedStatements.put(statementKey, statementInfo);
    }

    void addQueryRowCount(CON connectionKey, STMT statementKey, int rowCount) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)("Add query row count for connection key [" + connectionKey + "]"));
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            if (log.isTraceEnabled()) {
                log.trace((Object)"Connection is already closed");
            }
            return;
        }
        StatementInfo statementInfo = connectionInfo.nestedStatements.get(statementKey);
        SpanAndScope statementSpan = statementInfo.span;
        if (statementSpan != null) {
            AssertingSpan.of((DocumentedSpan)SleuthJdbcSpan.JDBC_QUERY_SPAN, (Span)statementSpan.getSpan()).tag((TagKey)SleuthJdbcSpan.QueryTags.ROW_COUNT, String.valueOf(rowCount));
        }
    }

    void afterQuery(CON connectionKey, STMT statementKey, String sql, @Nullable Throwable t) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)("After query for connection key [" + connectionKey + "]"));
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            if (log.isTraceEnabled()) {
                log.trace((Object)("Connection may be closed after statement preparation, but before statement execution. Current span is [" + this.getTracer().currentSpan() + "]"));
            }
            return;
        }
        StatementInfo statementInfo = connectionInfo.nestedStatements.get(statementKey);
        SpanAndScope statementSpan = statementInfo.span;
        if (statementSpan != null) {
            AssertingSpan.of((DocumentedSpan)SleuthJdbcSpan.JDBC_QUERY_SPAN, (Span)statementSpan.getSpan()).tag((TagKey)SleuthJdbcSpan.QueryTags.QUERY, sql).name(SPAN_NAME_PROVIDER.getSpanNameFor(sql));
            if (t != null) {
                statementSpan.getSpan().error(t);
            }
            if (log.isTraceEnabled()) {
                log.trace((Object)("Closing statement span [" + statementSpan + "] - current span is [" + this.getTracer().currentSpan() + "]"));
            }
            statementSpan.close();
            if (log.isTraceEnabled()) {
                log.trace((Object)("Current span [" + this.getTracer().currentSpan() + "]"));
            }
        }
    }

    void beforeResultSetNext(CON connectionKey, STMT statementKey, RS resultSetKey) {
        if (log.isTraceEnabled()) {
            log.trace((Object)"Before result set next");
        }
        if (!this.traceTypes.contains((Object)TraceType.FETCH)) {
            return;
        }
        ConnectionInfo connectionInfo = this.openConnections.get(connectionKey);
        if (connectionInfo == null) {
            if (log.isTraceEnabled()) {
                log.trace((Object)"No connection info, skipping");
            }
            return;
        }
        if (connectionInfo.nestedResultSetSpans.containsKey(resultSetKey)) {
            if (log.isTraceEnabled()) {
                log.trace((Object)"ResultSet span is already created");
            }
            return;
        }
        AssertingSpanBuilder resultSetSpanBuilder = AssertingSpanBuilder.of((DocumentedSpan)SleuthJdbcSpan.JDBC_RESULT_SET_SPAN, (Span.Builder)this.getTracer().spanBuilder()).name(SleuthJdbcSpan.JDBC_RESULT_SET_SPAN.getName());
        resultSetSpanBuilder.kind(Span.Kind.CLIENT);
        resultSetSpanBuilder.remoteServiceName(connectionInfo.remoteServiceName);
        if (connectionInfo.url != null) {
            resultSetSpanBuilder.remoteIpAndPort(connectionInfo.url.getHost(), connectionInfo.url.getPort());
        }
        AssertingSpan resultSetSpan = resultSetSpanBuilder.start();
        Tracer.SpanInScope scope = this.isCurrent(connectionInfo) ? this.getTracer().withSpan((Span)resultSetSpan) : null;
        SpanAndScope spanAndScope = new SpanAndScope((Span)resultSetSpan, scope);
        if (log.isTraceEnabled()) {
            log.trace((Object)("Started client result set span [" + resultSetSpan + "] - current span is [" + this.getTracer().currentSpan() + "]"));
        }
        connectionInfo.nestedResultSetSpans.put(resultSetKey, spanAndScope);
        StatementInfo statementInfo = connectionInfo.nestedStatements.get(statementKey);
        if (statementInfo != null) {
            statementInfo.nestedResultSetSpans.put(resultSetKey, spanAndScope);
        }
    }

    void afterStatementClose(CON connectionKey, STMT statementKey) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)"After statement close");
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            return;
        }
        StatementInfo statementInfo = connectionInfo.nestedStatements.remove(statementKey);
        if (statementInfo != null) {
            statementInfo.nestedResultSetSpans.forEach((resultSetKey, span) -> {
                connectionInfo.nestedResultSetSpans.remove(resultSetKey);
                if (log.isTraceEnabled()) {
                    log.trace((Object)("Closing span after statement close [" + span.getSpan() + "] - current span is [" + this.getTracer().currentSpan() + "]"));
                }
                span.close();
                if (log.isTraceEnabled()) {
                    log.trace((Object)("Current span [" + this.getTracer().currentSpan() + "]"));
                }
            });
            statementInfo.nestedResultSetSpans.clear();
        }
    }

    void afterResultSetClose(CON connectionKey, RS resultSetKey, int rowCount, @Nullable Throwable t) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)"After result set close");
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            return;
        }
        SpanAndScope resultSetSpan = connectionInfo.nestedResultSetSpans.remove(resultSetKey);
        if (resultSetSpan == null) {
            return;
        }
        if (rowCount != -1) {
            AssertingSpan.of((DocumentedSpan)SleuthJdbcSpan.JDBC_RESULT_SET_SPAN, (Span)resultSetSpan.getSpan()).tag((TagKey)SleuthJdbcSpan.QueryTags.ROW_COUNT, String.valueOf(rowCount));
        }
        if (t != null) {
            resultSetSpan.getSpan().error(t);
        }
        if (log.isTraceEnabled()) {
            log.trace((Object)("Closing client result set span [" + resultSetSpan + "] - current span is [" + this.getTracer().currentSpan() + "]"));
        }
        resultSetSpan.close();
        if (log.isTraceEnabled()) {
            log.trace((Object)("Current span [" + this.getTracer().currentSpan() + "]"));
        }
    }

    void afterCommit(CON connectionKey, @Nullable Throwable t) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)"After commit");
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            return;
        }
        SpanAndScope connectionSpan = connectionInfo.span;
        if (connectionSpan != null) {
            if (t != null) {
                connectionSpan.getSpan().error(t);
            }
            AssertingSpan.of((DocumentedSpan)SleuthJdbcSpan.JDBC_QUERY_SPAN, (Span)connectionSpan.getSpan()).event((EventValue)SleuthJdbcSpan.QueryEvents.COMMIT);
        }
    }

    void afterRollback(CON connectionKey, @Nullable Throwable t) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)"After rollback");
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            return;
        }
        SpanAndScope connectionSpan = connectionInfo.span;
        if (connectionSpan != null) {
            if (t != null) {
                connectionSpan.getSpan().error(t);
            } else {
                connectionSpan.getSpan().error((Throwable)new JdbcException("Transaction rolled back"));
            }
            AssertingSpan.of((DocumentedSpan)SleuthJdbcSpan.JDBC_QUERY_SPAN, (Span)connectionSpan.getSpan()).event((EventValue)SleuthJdbcSpan.QueryEvents.ROLLBACK);
        }
    }

    void afterConnectionClose(CON connectionKey, @Nullable Throwable t) {
        SpanAndScope connectionSpan;
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)("After connection close with key [" + connectionKey + "]"));
        }
        if (this.isCurrent(connectionInfo = this.openConnections.remove(connectionKey))) {
            this.currentConnection.set(null);
        }
        if (connectionInfo == null) {
            return;
        }
        connectionInfo.nestedResultSetSpans.values().forEach(SpanAndScope::close);
        connectionInfo.nestedStatements.values().forEach(statementInfo -> {
            SpanAndScope statementSpan = statementInfo.span;
            if (statementSpan != null) {
                statementSpan.close();
            }
        });
        if (log.isTraceEnabled()) {
            log.trace((Object)("Current span after closing statements [" + this.getTracer().currentSpan() + "]"));
        }
        if ((connectionSpan = connectionInfo.span) != null) {
            if (t != null) {
                connectionSpan.getSpan().error(t);
            }
            if (log.isTraceEnabled()) {
                log.trace((Object)("Closing span after connection close [" + connectionSpan.getSpan() + "] - current span is [" + this.getTracer().currentSpan() + "]"));
            }
            connectionSpan.close();
            if (log.isTraceEnabled()) {
                log.trace((Object)("Current span [" + this.getTracer().currentSpan() + "]"));
            }
        }
    }

    private void parseAndSetServerIpAndPort(ConnectionInfo connectionInfo, Connection connection, String dataSourceName) {
        URI url = null;
        String remoteServiceName = "";
        try {
            String databaseName;
            String parsedServiceName;
            String urlAsString = connection.getMetaData().getURL().substring(5);
            url = URI.create(urlAsString.replace(" ", ""));
            Matcher matcher = URL_SERVICE_NAME_FINDER.matcher(url.toString());
            if (matcher.find() && matcher.groupCount() == 1 && (parsedServiceName = matcher.group(1)) != null && !parsedServiceName.isEmpty()) {
                remoteServiceName = parsedServiceName;
            }
            if (!StringUtils.hasText((String)remoteServiceName) && (databaseName = connection.getCatalog()) != null && !databaseName.isEmpty()) {
                remoteServiceName = databaseName;
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        connectionInfo.url = url;
        connectionInfo.remoteServiceName = StringUtils.hasText((String)remoteServiceName) ? remoteServiceName : dataSourceName;
    }

    private Tracer getTracer() {
        if (this.tracer == null) {
            this.tracer = (Tracer)this.beanFactory.getBean(Tracer.class);
        }
        return this.tracer;
    }

    private static final class JdbcException
    extends RuntimeException {
        JdbcException(String message) {
            super(message);
        }
    }

    private final class StatementInfo {
        final SpanAndScope span;
        final Map<RS, SpanAndScope> nestedResultSetSpans = new ConcurrentHashMap();

        StatementInfo(SpanAndScope span) {
            this.span = span;
        }
    }

    private final class ConnectionInfo {
        final SpanAndScope span;
        final Map<STMT, StatementInfo> nestedStatements = new ConcurrentHashMap();
        final Map<RS, SpanAndScope> nestedResultSetSpans = new ConcurrentHashMap();
        URI url;
        String remoteServiceName;

        ConnectionInfo(SpanAndScope span) {
            this.span = span;
        }
    }
}

