/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.memorydb.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CopySnapshotRequest extends MemoryDbRequest implements
        ToCopyableBuilder<CopySnapshotRequest.Builder, CopySnapshotRequest> {
    private static final SdkField<String> SOURCE_SNAPSHOT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceSnapshotName").getter(getter(CopySnapshotRequest::sourceSnapshotName))
            .setter(setter(Builder::sourceSnapshotName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceSnapshotName").build())
            .build();

    private static final SdkField<String> TARGET_SNAPSHOT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetSnapshotName").getter(getter(CopySnapshotRequest::targetSnapshotName))
            .setter(setter(Builder::targetSnapshotName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetSnapshotName").build())
            .build();

    private static final SdkField<String> TARGET_BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetBucket").getter(getter(CopySnapshotRequest::targetBucket)).setter(setter(Builder::targetBucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetBucket").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(CopySnapshotRequest::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CopySnapshotRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_SNAPSHOT_NAME_FIELD,
            TARGET_SNAPSHOT_NAME_FIELD, TARGET_BUCKET_FIELD, KMS_KEY_ID_FIELD, TAGS_FIELD));

    private final String sourceSnapshotName;

    private final String targetSnapshotName;

    private final String targetBucket;

    private final String kmsKeyId;

    private final List<Tag> tags;

    private CopySnapshotRequest(BuilderImpl builder) {
        super(builder);
        this.sourceSnapshotName = builder.sourceSnapshotName;
        this.targetSnapshotName = builder.targetSnapshotName;
        this.targetBucket = builder.targetBucket;
        this.kmsKeyId = builder.kmsKeyId;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of an existing snapshot from which to make a copy.
     * </p>
     * 
     * @return The name of an existing snapshot from which to make a copy.
     */
    public final String sourceSnapshotName() {
        return sourceSnapshotName;
    }

    /**
     * <p>
     * A name for the snapshot copy. MemoryDB does not permit overwriting a snapshot, therefore this name must be unique
     * within its context - MemoryDB or an Amazon S3 bucket if exporting.
     * </p>
     * 
     * @return A name for the snapshot copy. MemoryDB does not permit overwriting a snapshot, therefore this name must
     *         be unique within its context - MemoryDB or an Amazon S3 bucket if exporting.
     */
    public final String targetSnapshotName() {
        return targetSnapshotName;
    }

    /**
     * <p>
     * The Amazon S3 bucket to which the snapshot is exported. This parameter is used only when exporting a snapshot for
     * external access. When using this parameter to export a snapshot, be sure MemoryDB has the needed permissions to
     * this S3 bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/snapshots-exporting.html">Step 2: Grant MemoryDB
     * Access to Your Amazon S3 Bucket</a>.
     * </p>
     * 
     * @return The Amazon S3 bucket to which the snapshot is exported. This parameter is used only when exporting a
     *         snapshot for external access. When using this parameter to export a snapshot, be sure MemoryDB has the
     *         needed permissions to this S3 bucket. For more information, see <a
     *         href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/snapshots-exporting.html">Step 2: Grant
     *         MemoryDB Access to Your Amazon S3 Bucket</a>.
     */
    public final String targetBucket() {
        return targetBucket;
    }

    /**
     * <p>
     * The ID of the KMS key used to encrypt the target snapshot.
     * </p>
     * 
     * @return The ID of the KMS key used to encrypt the target snapshot.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags to be added to this resource. A tag is a key-value pair. A tag key must be accompanied by a tag
     * value, although null is accepted.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags to be added to this resource. A tag is a key-value pair. A tag key must be accompanied by
     *         a tag value, although null is accepted.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(sourceSnapshotName());
        hashCode = 31 * hashCode + Objects.hashCode(targetSnapshotName());
        hashCode = 31 * hashCode + Objects.hashCode(targetBucket());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CopySnapshotRequest)) {
            return false;
        }
        CopySnapshotRequest other = (CopySnapshotRequest) obj;
        return Objects.equals(sourceSnapshotName(), other.sourceSnapshotName())
                && Objects.equals(targetSnapshotName(), other.targetSnapshotName())
                && Objects.equals(targetBucket(), other.targetBucket()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CopySnapshotRequest").add("SourceSnapshotName", sourceSnapshotName())
                .add("TargetSnapshotName", targetSnapshotName()).add("TargetBucket", targetBucket()).add("KmsKeyId", kmsKeyId())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SourceSnapshotName":
            return Optional.ofNullable(clazz.cast(sourceSnapshotName()));
        case "TargetSnapshotName":
            return Optional.ofNullable(clazz.cast(targetSnapshotName()));
        case "TargetBucket":
            return Optional.ofNullable(clazz.cast(targetBucket()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CopySnapshotRequest, T> g) {
        return obj -> g.apply((CopySnapshotRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends MemoryDbRequest.Builder, SdkPojo, CopyableBuilder<Builder, CopySnapshotRequest> {
        /**
         * <p>
         * The name of an existing snapshot from which to make a copy.
         * </p>
         * 
         * @param sourceSnapshotName
         *        The name of an existing snapshot from which to make a copy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceSnapshotName(String sourceSnapshotName);

        /**
         * <p>
         * A name for the snapshot copy. MemoryDB does not permit overwriting a snapshot, therefore this name must be
         * unique within its context - MemoryDB or an Amazon S3 bucket if exporting.
         * </p>
         * 
         * @param targetSnapshotName
         *        A name for the snapshot copy. MemoryDB does not permit overwriting a snapshot, therefore this name
         *        must be unique within its context - MemoryDB or an Amazon S3 bucket if exporting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetSnapshotName(String targetSnapshotName);

        /**
         * <p>
         * The Amazon S3 bucket to which the snapshot is exported. This parameter is used only when exporting a snapshot
         * for external access. When using this parameter to export a snapshot, be sure MemoryDB has the needed
         * permissions to this S3 bucket. For more information, see <a
         * href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/snapshots-exporting.html">Step 2: Grant MemoryDB
         * Access to Your Amazon S3 Bucket</a>.
         * </p>
         * 
         * @param targetBucket
         *        The Amazon S3 bucket to which the snapshot is exported. This parameter is used only when exporting a
         *        snapshot for external access. When using this parameter to export a snapshot, be sure MemoryDB has the
         *        needed permissions to this S3 bucket. For more information, see <a
         *        href="https://docs.aws.amazon.com/MemoryDB/latest/devguide/snapshots-exporting.html">Step 2: Grant
         *        MemoryDB Access to Your Amazon S3 Bucket</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetBucket(String targetBucket);

        /**
         * <p>
         * The ID of the KMS key used to encrypt the target snapshot.
         * </p>
         * 
         * @param kmsKeyId
         *        The ID of the KMS key used to encrypt the target snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * A list of tags to be added to this resource. A tag is a key-value pair. A tag key must be accompanied by a
         * tag value, although null is accepted.
         * </p>
         * 
         * @param tags
         *        A list of tags to be added to this resource. A tag is a key-value pair. A tag key must be accompanied
         *        by a tag value, although null is accepted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags to be added to this resource. A tag is a key-value pair. A tag key must be accompanied by a
         * tag value, although null is accepted.
         * </p>
         * 
         * @param tags
         *        A list of tags to be added to this resource. A tag is a key-value pair. A tag key must be accompanied
         *        by a tag value, although null is accepted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags to be added to this resource. A tag is a key-value pair. A tag key must be accompanied by a
         * tag value, although null is accepted.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.memorydb.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.memorydb.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.memorydb.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.memorydb.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends MemoryDbRequest.BuilderImpl implements Builder {
        private String sourceSnapshotName;

        private String targetSnapshotName;

        private String targetBucket;

        private String kmsKeyId;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CopySnapshotRequest model) {
            super(model);
            sourceSnapshotName(model.sourceSnapshotName);
            targetSnapshotName(model.targetSnapshotName);
            targetBucket(model.targetBucket);
            kmsKeyId(model.kmsKeyId);
            tags(model.tags);
        }

        public final String getSourceSnapshotName() {
            return sourceSnapshotName;
        }

        public final void setSourceSnapshotName(String sourceSnapshotName) {
            this.sourceSnapshotName = sourceSnapshotName;
        }

        @Override
        public final Builder sourceSnapshotName(String sourceSnapshotName) {
            this.sourceSnapshotName = sourceSnapshotName;
            return this;
        }

        public final String getTargetSnapshotName() {
            return targetSnapshotName;
        }

        public final void setTargetSnapshotName(String targetSnapshotName) {
            this.targetSnapshotName = targetSnapshotName;
        }

        @Override
        public final Builder targetSnapshotName(String targetSnapshotName) {
            this.targetSnapshotName = targetSnapshotName;
            return this;
        }

        public final String getTargetBucket() {
            return targetBucket;
        }

        public final void setTargetBucket(String targetBucket) {
            this.targetBucket = targetBucket;
        }

        @Override
        public final Builder targetBucket(String targetBucket) {
            this.targetBucket = targetBucket;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CopySnapshotRequest build() {
            return new CopySnapshotRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
