/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The analysis result for Network Firewall's stateless rule group analyzer. Every time you call <a>CreateRuleGroup</a>,
 * <a>UpdateRuleGroup</a>, or <a>DescribeRuleGroup</a> on a stateless rule group, Network Firewall analyzes the
 * stateless rule groups in your account and identifies the rules that might adversely effect your firewall's
 * functionality. For example, if Network Firewall detects a rule that's routing traffic asymmetrically, which impacts
 * the service's ability to properly process traffic, the service includes the rule in a list of analysis results.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AnalysisResult implements SdkPojo, Serializable, ToCopyableBuilder<AnalysisResult.Builder, AnalysisResult> {
    private static final SdkField<List<String>> IDENTIFIED_RULE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("IdentifiedRuleIds")
            .getter(getter(AnalysisResult::identifiedRuleIds))
            .setter(setter(Builder::identifiedRuleIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifiedRuleIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> IDENTIFIED_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IdentifiedType").getter(getter(AnalysisResult::identifiedTypeAsString))
            .setter(setter(Builder::identifiedType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifiedType").build()).build();

    private static final SdkField<String> ANALYSIS_DETAIL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AnalysisDetail").getter(getter(AnalysisResult::analysisDetail)).setter(setter(Builder::analysisDetail))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnalysisDetail").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IDENTIFIED_RULE_IDS_FIELD,
            IDENTIFIED_TYPE_FIELD, ANALYSIS_DETAIL_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> identifiedRuleIds;

    private final String identifiedType;

    private final String analysisDetail;

    private AnalysisResult(BuilderImpl builder) {
        this.identifiedRuleIds = builder.identifiedRuleIds;
        this.identifiedType = builder.identifiedType;
        this.analysisDetail = builder.analysisDetail;
    }

    /**
     * For responses, this returns true if the service returned a value for the IdentifiedRuleIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIdentifiedRuleIds() {
        return identifiedRuleIds != null && !(identifiedRuleIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The priority number of the stateless rules identified in the analysis.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIdentifiedRuleIds} method.
     * </p>
     * 
     * @return The priority number of the stateless rules identified in the analysis.
     */
    public final List<String> identifiedRuleIds() {
        return identifiedRuleIds;
    }

    /**
     * <p>
     * The types of rule configurations that Network Firewall analyzes your rule groups for. Network Firewall analyzes
     * stateless rule groups for the following types of rule configurations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>STATELESS_RULE_FORWARDING_ASYMMETRICALLY</code>
     * </p>
     * <p>
     * Cause: One or more stateless rules with the action <code>pass</code> or <code>forward</code> are forwarding
     * traffic asymmetrically. Specifically, the rule's set of source IP addresses or their associated port numbers,
     * don't match the set of destination IP addresses or their associated port numbers.
     * </p>
     * <p>
     * To mitigate: Make sure that there's an existing return path. For example, if the rule allows traffic from source
     * 10.1.0.0/24 to destination 20.1.0.0/24, you should allow return traffic from source 20.1.0.0/24 to destination
     * 10.1.0.0/24.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STATELESS_RULE_CONTAINS_TCP_FLAGS</code>
     * </p>
     * <p>
     * Cause: At least one stateless rule with the action <code>pass</code> or<code>forward</code> contains TCP flags
     * that are inconsistent in the forward and return directions.
     * </p>
     * <p>
     * To mitigate: Prevent asymmetric routing issues caused by TCP flags by following these actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Remove unnecessary TCP flag inspections from the rules.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you need to inspect TCP flags, check that the rules correctly account for changes in TCP flags throughout the
     * TCP connection cycle, for example <code>SYN</code> and <code>ACK</code> flags used in a 3-way TCP handshake.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #identifiedType}
     * will return {@link IdentifiedType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #identifiedTypeAsString}.
     * </p>
     * 
     * @return The types of rule configurations that Network Firewall analyzes your rule groups for. Network Firewall
     *         analyzes stateless rule groups for the following types of rule configurations:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>STATELESS_RULE_FORWARDING_ASYMMETRICALLY</code>
     *         </p>
     *         <p>
     *         Cause: One or more stateless rules with the action <code>pass</code> or <code>forward</code> are
     *         forwarding traffic asymmetrically. Specifically, the rule's set of source IP addresses or their
     *         associated port numbers, don't match the set of destination IP addresses or their associated port
     *         numbers.
     *         </p>
     *         <p>
     *         To mitigate: Make sure that there's an existing return path. For example, if the rule allows traffic from
     *         source 10.1.0.0/24 to destination 20.1.0.0/24, you should allow return traffic from source 20.1.0.0/24 to
     *         destination 10.1.0.0/24.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STATELESS_RULE_CONTAINS_TCP_FLAGS</code>
     *         </p>
     *         <p>
     *         Cause: At least one stateless rule with the action <code>pass</code> or<code>forward</code> contains TCP
     *         flags that are inconsistent in the forward and return directions.
     *         </p>
     *         <p>
     *         To mitigate: Prevent asymmetric routing issues caused by TCP flags by following these actions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Remove unnecessary TCP flag inspections from the rules.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you need to inspect TCP flags, check that the rules correctly account for changes in TCP flags
     *         throughout the TCP connection cycle, for example <code>SYN</code> and <code>ACK</code> flags used in a
     *         3-way TCP handshake.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see IdentifiedType
     */
    public final IdentifiedType identifiedType() {
        return IdentifiedType.fromValue(identifiedType);
    }

    /**
     * <p>
     * The types of rule configurations that Network Firewall analyzes your rule groups for. Network Firewall analyzes
     * stateless rule groups for the following types of rule configurations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>STATELESS_RULE_FORWARDING_ASYMMETRICALLY</code>
     * </p>
     * <p>
     * Cause: One or more stateless rules with the action <code>pass</code> or <code>forward</code> are forwarding
     * traffic asymmetrically. Specifically, the rule's set of source IP addresses or their associated port numbers,
     * don't match the set of destination IP addresses or their associated port numbers.
     * </p>
     * <p>
     * To mitigate: Make sure that there's an existing return path. For example, if the rule allows traffic from source
     * 10.1.0.0/24 to destination 20.1.0.0/24, you should allow return traffic from source 20.1.0.0/24 to destination
     * 10.1.0.0/24.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STATELESS_RULE_CONTAINS_TCP_FLAGS</code>
     * </p>
     * <p>
     * Cause: At least one stateless rule with the action <code>pass</code> or<code>forward</code> contains TCP flags
     * that are inconsistent in the forward and return directions.
     * </p>
     * <p>
     * To mitigate: Prevent asymmetric routing issues caused by TCP flags by following these actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Remove unnecessary TCP flag inspections from the rules.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you need to inspect TCP flags, check that the rules correctly account for changes in TCP flags throughout the
     * TCP connection cycle, for example <code>SYN</code> and <code>ACK</code> flags used in a 3-way TCP handshake.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #identifiedType}
     * will return {@link IdentifiedType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #identifiedTypeAsString}.
     * </p>
     * 
     * @return The types of rule configurations that Network Firewall analyzes your rule groups for. Network Firewall
     *         analyzes stateless rule groups for the following types of rule configurations:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>STATELESS_RULE_FORWARDING_ASYMMETRICALLY</code>
     *         </p>
     *         <p>
     *         Cause: One or more stateless rules with the action <code>pass</code> or <code>forward</code> are
     *         forwarding traffic asymmetrically. Specifically, the rule's set of source IP addresses or their
     *         associated port numbers, don't match the set of destination IP addresses or their associated port
     *         numbers.
     *         </p>
     *         <p>
     *         To mitigate: Make sure that there's an existing return path. For example, if the rule allows traffic from
     *         source 10.1.0.0/24 to destination 20.1.0.0/24, you should allow return traffic from source 20.1.0.0/24 to
     *         destination 10.1.0.0/24.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STATELESS_RULE_CONTAINS_TCP_FLAGS</code>
     *         </p>
     *         <p>
     *         Cause: At least one stateless rule with the action <code>pass</code> or<code>forward</code> contains TCP
     *         flags that are inconsistent in the forward and return directions.
     *         </p>
     *         <p>
     *         To mitigate: Prevent asymmetric routing issues caused by TCP flags by following these actions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Remove unnecessary TCP flag inspections from the rules.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you need to inspect TCP flags, check that the rules correctly account for changes in TCP flags
     *         throughout the TCP connection cycle, for example <code>SYN</code> and <code>ACK</code> flags used in a
     *         3-way TCP handshake.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see IdentifiedType
     */
    public final String identifiedTypeAsString() {
        return identifiedType;
    }

    /**
     * <p>
     * Provides analysis details for the identified rule.
     * </p>
     * 
     * @return Provides analysis details for the identified rule.
     */
    public final String analysisDetail() {
        return analysisDetail;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasIdentifiedRuleIds() ? identifiedRuleIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(identifiedTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(analysisDetail());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AnalysisResult)) {
            return false;
        }
        AnalysisResult other = (AnalysisResult) obj;
        return hasIdentifiedRuleIds() == other.hasIdentifiedRuleIds()
                && Objects.equals(identifiedRuleIds(), other.identifiedRuleIds())
                && Objects.equals(identifiedTypeAsString(), other.identifiedTypeAsString())
                && Objects.equals(analysisDetail(), other.analysisDetail());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AnalysisResult").add("IdentifiedRuleIds", hasIdentifiedRuleIds() ? identifiedRuleIds() : null)
                .add("IdentifiedType", identifiedTypeAsString()).add("AnalysisDetail", analysisDetail()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IdentifiedRuleIds":
            return Optional.ofNullable(clazz.cast(identifiedRuleIds()));
        case "IdentifiedType":
            return Optional.ofNullable(clazz.cast(identifiedTypeAsString()));
        case "AnalysisDetail":
            return Optional.ofNullable(clazz.cast(analysisDetail()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AnalysisResult, T> g) {
        return obj -> g.apply((AnalysisResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AnalysisResult> {
        /**
         * <p>
         * The priority number of the stateless rules identified in the analysis.
         * </p>
         * 
         * @param identifiedRuleIds
         *        The priority number of the stateless rules identified in the analysis.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifiedRuleIds(Collection<String> identifiedRuleIds);

        /**
         * <p>
         * The priority number of the stateless rules identified in the analysis.
         * </p>
         * 
         * @param identifiedRuleIds
         *        The priority number of the stateless rules identified in the analysis.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifiedRuleIds(String... identifiedRuleIds);

        /**
         * <p>
         * The types of rule configurations that Network Firewall analyzes your rule groups for. Network Firewall
         * analyzes stateless rule groups for the following types of rule configurations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>STATELESS_RULE_FORWARDING_ASYMMETRICALLY</code>
         * </p>
         * <p>
         * Cause: One or more stateless rules with the action <code>pass</code> or <code>forward</code> are forwarding
         * traffic asymmetrically. Specifically, the rule's set of source IP addresses or their associated port numbers,
         * don't match the set of destination IP addresses or their associated port numbers.
         * </p>
         * <p>
         * To mitigate: Make sure that there's an existing return path. For example, if the rule allows traffic from
         * source 10.1.0.0/24 to destination 20.1.0.0/24, you should allow return traffic from source 20.1.0.0/24 to
         * destination 10.1.0.0/24.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STATELESS_RULE_CONTAINS_TCP_FLAGS</code>
         * </p>
         * <p>
         * Cause: At least one stateless rule with the action <code>pass</code> or<code>forward</code> contains TCP
         * flags that are inconsistent in the forward and return directions.
         * </p>
         * <p>
         * To mitigate: Prevent asymmetric routing issues caused by TCP flags by following these actions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Remove unnecessary TCP flag inspections from the rules.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you need to inspect TCP flags, check that the rules correctly account for changes in TCP flags throughout
         * the TCP connection cycle, for example <code>SYN</code> and <code>ACK</code> flags used in a 3-way TCP
         * handshake.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param identifiedType
         *        The types of rule configurations that Network Firewall analyzes your rule groups for. Network Firewall
         *        analyzes stateless rule groups for the following types of rule configurations:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>STATELESS_RULE_FORWARDING_ASYMMETRICALLY</code>
         *        </p>
         *        <p>
         *        Cause: One or more stateless rules with the action <code>pass</code> or <code>forward</code> are
         *        forwarding traffic asymmetrically. Specifically, the rule's set of source IP addresses or their
         *        associated port numbers, don't match the set of destination IP addresses or their associated port
         *        numbers.
         *        </p>
         *        <p>
         *        To mitigate: Make sure that there's an existing return path. For example, if the rule allows traffic
         *        from source 10.1.0.0/24 to destination 20.1.0.0/24, you should allow return traffic from source
         *        20.1.0.0/24 to destination 10.1.0.0/24.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STATELESS_RULE_CONTAINS_TCP_FLAGS</code>
         *        </p>
         *        <p>
         *        Cause: At least one stateless rule with the action <code>pass</code> or<code>forward</code> contains
         *        TCP flags that are inconsistent in the forward and return directions.
         *        </p>
         *        <p>
         *        To mitigate: Prevent asymmetric routing issues caused by TCP flags by following these actions:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Remove unnecessary TCP flag inspections from the rules.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you need to inspect TCP flags, check that the rules correctly account for changes in TCP flags
         *        throughout the TCP connection cycle, for example <code>SYN</code> and <code>ACK</code> flags used in a
         *        3-way TCP handshake.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see IdentifiedType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IdentifiedType
         */
        Builder identifiedType(String identifiedType);

        /**
         * <p>
         * The types of rule configurations that Network Firewall analyzes your rule groups for. Network Firewall
         * analyzes stateless rule groups for the following types of rule configurations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>STATELESS_RULE_FORWARDING_ASYMMETRICALLY</code>
         * </p>
         * <p>
         * Cause: One or more stateless rules with the action <code>pass</code> or <code>forward</code> are forwarding
         * traffic asymmetrically. Specifically, the rule's set of source IP addresses or their associated port numbers,
         * don't match the set of destination IP addresses or their associated port numbers.
         * </p>
         * <p>
         * To mitigate: Make sure that there's an existing return path. For example, if the rule allows traffic from
         * source 10.1.0.0/24 to destination 20.1.0.0/24, you should allow return traffic from source 20.1.0.0/24 to
         * destination 10.1.0.0/24.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STATELESS_RULE_CONTAINS_TCP_FLAGS</code>
         * </p>
         * <p>
         * Cause: At least one stateless rule with the action <code>pass</code> or<code>forward</code> contains TCP
         * flags that are inconsistent in the forward and return directions.
         * </p>
         * <p>
         * To mitigate: Prevent asymmetric routing issues caused by TCP flags by following these actions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Remove unnecessary TCP flag inspections from the rules.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you need to inspect TCP flags, check that the rules correctly account for changes in TCP flags throughout
         * the TCP connection cycle, for example <code>SYN</code> and <code>ACK</code> flags used in a 3-way TCP
         * handshake.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param identifiedType
         *        The types of rule configurations that Network Firewall analyzes your rule groups for. Network Firewall
         *        analyzes stateless rule groups for the following types of rule configurations:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>STATELESS_RULE_FORWARDING_ASYMMETRICALLY</code>
         *        </p>
         *        <p>
         *        Cause: One or more stateless rules with the action <code>pass</code> or <code>forward</code> are
         *        forwarding traffic asymmetrically. Specifically, the rule's set of source IP addresses or their
         *        associated port numbers, don't match the set of destination IP addresses or their associated port
         *        numbers.
         *        </p>
         *        <p>
         *        To mitigate: Make sure that there's an existing return path. For example, if the rule allows traffic
         *        from source 10.1.0.0/24 to destination 20.1.0.0/24, you should allow return traffic from source
         *        20.1.0.0/24 to destination 10.1.0.0/24.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STATELESS_RULE_CONTAINS_TCP_FLAGS</code>
         *        </p>
         *        <p>
         *        Cause: At least one stateless rule with the action <code>pass</code> or<code>forward</code> contains
         *        TCP flags that are inconsistent in the forward and return directions.
         *        </p>
         *        <p>
         *        To mitigate: Prevent asymmetric routing issues caused by TCP flags by following these actions:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Remove unnecessary TCP flag inspections from the rules.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you need to inspect TCP flags, check that the rules correctly account for changes in TCP flags
         *        throughout the TCP connection cycle, for example <code>SYN</code> and <code>ACK</code> flags used in a
         *        3-way TCP handshake.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see IdentifiedType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IdentifiedType
         */
        Builder identifiedType(IdentifiedType identifiedType);

        /**
         * <p>
         * Provides analysis details for the identified rule.
         * </p>
         * 
         * @param analysisDetail
         *        Provides analysis details for the identified rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analysisDetail(String analysisDetail);
    }

    static final class BuilderImpl implements Builder {
        private List<String> identifiedRuleIds = DefaultSdkAutoConstructList.getInstance();

        private String identifiedType;

        private String analysisDetail;

        private BuilderImpl() {
        }

        private BuilderImpl(AnalysisResult model) {
            identifiedRuleIds(model.identifiedRuleIds);
            identifiedType(model.identifiedType);
            analysisDetail(model.analysisDetail);
        }

        public final Collection<String> getIdentifiedRuleIds() {
            if (identifiedRuleIds instanceof SdkAutoConstructList) {
                return null;
            }
            return identifiedRuleIds;
        }

        public final void setIdentifiedRuleIds(Collection<String> identifiedRuleIds) {
            this.identifiedRuleIds = RuleIdListCopier.copy(identifiedRuleIds);
        }

        @Override
        public final Builder identifiedRuleIds(Collection<String> identifiedRuleIds) {
            this.identifiedRuleIds = RuleIdListCopier.copy(identifiedRuleIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder identifiedRuleIds(String... identifiedRuleIds) {
            identifiedRuleIds(Arrays.asList(identifiedRuleIds));
            return this;
        }

        public final String getIdentifiedType() {
            return identifiedType;
        }

        public final void setIdentifiedType(String identifiedType) {
            this.identifiedType = identifiedType;
        }

        @Override
        public final Builder identifiedType(String identifiedType) {
            this.identifiedType = identifiedType;
            return this;
        }

        @Override
        public final Builder identifiedType(IdentifiedType identifiedType) {
            this.identifiedType(identifiedType == null ? null : identifiedType.toString());
            return this;
        }

        public final String getAnalysisDetail() {
            return analysisDetail;
        }

        public final void setAnalysisDetail(String analysisDetail) {
            this.analysisDetail = analysisDetail;
        }

        @Override
        public final Builder analysisDetail(String analysisDetail) {
            this.analysisDetail = analysisDetail;
            return this;
        }

        @Override
        public AnalysisResult build() {
            return new AnalysisResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
