package ai.digital.config.server.api

import ai.digital.config.{ConfigFileFilter, ConfigFileMapping}
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.cloud.config.environment.Environment
import org.springframework.cloud.config.server.environment.InvalidEnvironmentRequestException
import org.springframework.cloud.config.server.support.PathUtils
import org.springframework.http.MediaType
import org.springframework.web.bind.annotation._

import scala.jdk.CollectionConverters._

@RestController
@RequestMapping(path = Array("${spring.cloud.config.server.prefix:}"))
class CentralConfigurationPropertiesController(@Qualifier("environmentConfigurationPropertiesService")
                                               val centralConfigurationPropertiesService: ConfigurationPropertiesService) {

  @PostMapping(
    path = Array(PathConstants.PROPERTIES_PATH),
    consumes = Array(MediaType.APPLICATION_JSON_VALUE),
    produces = Array(MediaType.APPLICATION_JSON_VALUE)
  )
  def propertiesStoreDefaultLabel(@PathVariable name: String,
                                  @PathVariable profiles: String,
                                  @RequestBody requestBody: PropertiesStoreRequestBody): PropertiesStoreResponse =
    propertiesStoreDefaultLabel(name, profiles, null, requestBody)

  @PostMapping(
    path = Array(PathConstants.PROPERTIES_PATH_WITH_LABEL),
    consumes = Array(MediaType.APPLICATION_JSON_VALUE),
    produces = Array(MediaType.APPLICATION_JSON_VALUE)
  )
  def propertiesStoreDefaultLabel(@PathVariable name: String,
                                  @PathVariable profiles: String,
                                  @PathVariable label: String,
                                  @RequestBody requestBody: PropertiesStoreRequestBody): PropertiesStoreResponse = {
    normalize(name)
    normalize(profiles)
    normalize(label)

    PropertiesStoreResponse(
      centralConfigurationPropertiesService.propertiesStore(
        ConfigFileFilter(name, profiles, label),
        requestBody.header,
        requestBody.properties.asScala.map(prop => (prop.key, prop.value)),
        ConfigFileMapping(
          Option(requestBody.configFileToPropertyKeysMapping).map(_.asScala.toMap).getOrElse(Map.empty),
          Option(requestBody.defaultConfigFile)
        )
      ).asJava
    )
  }

  @PutMapping(
    path = Array(PathConstants.PROPERTIES_PATH),
    consumes = Array(MediaType.APPLICATION_JSON_VALUE),
    produces = Array(MediaType.APPLICATION_JSON_VALUE)
  )
  def propertiesUpdateDefaultLabel(@PathVariable name: String,
                                   @PathVariable profiles: String,
                                   @RequestBody requestBody: PropertiesUpdateRequestBody): PropertiesUpdateResponse =
    propertiesUpdate(name, profiles, null, requestBody)

  @PutMapping(
    path = Array(PathConstants.PROPERTIES_PATH_WITH_LABEL),
    consumes = Array(MediaType.APPLICATION_JSON_VALUE),
    produces = Array(MediaType.APPLICATION_JSON_VALUE)
  )
  def propertiesUpdate(@PathVariable name: String,
                       @PathVariable profiles: String,
                       @PathVariable label: String,
                       @RequestBody requestBody: PropertiesUpdateRequestBody): PropertiesUpdateResponse = {
    normalize(name)
    normalize(profiles)
    normalize(label)

    PropertiesUpdateResponse(
      centralConfigurationPropertiesService.propertiesUpdate(
        ConfigFileFilter(name, profiles, label),
        requestBody.properties.asScala.toMap,
        ConfigFileMapping(
          Option(requestBody.configFileToPropertyKeysMapping).map(_.asScala.toMap).getOrElse(Map.empty),
          Option(requestBody.defaultConfigFile)
        )
      ).asJava
    )
  }

  @DeleteMapping(
    path = Array(PathConstants.PROPERTIES_PATH),
    consumes = Array(MediaType.APPLICATION_JSON_VALUE),
    produces = Array(MediaType.APPLICATION_JSON_VALUE)
  )
  def propertiesDeleteDefaultLabel(@PathVariable name: String,
                                   @PathVariable profiles: String,
                                   @RequestBody requestBody: PropertiesDeleteRequestBody): PropertiesDeleteResponse =
    propertiesDelete(name, profiles, null, requestBody)

  @DeleteMapping(
    path = Array(PathConstants.PROPERTIES_PATH_WITH_LABEL),
    consumes = Array(MediaType.APPLICATION_JSON_VALUE),
    produces = Array(MediaType.APPLICATION_JSON_VALUE)
  )
  def propertiesDelete(@PathVariable name: String,
                       @PathVariable profiles: String,
                       @PathVariable label: String,
                       @RequestBody requestBody: PropertiesDeleteRequestBody): PropertiesDeleteResponse = {
    normalize(name)
    normalize(profiles)
    normalize(label)
    PropertiesDeleteResponse(
      centralConfigurationPropertiesService.propertiesDelete(
        ConfigFileFilter(name, profiles, label),
        requestBody.keys.asScala.toSet,
        ConfigFileMapping(
          Option(requestBody.configFileToPropertyKeysMapping).map(_.asScala.toMap).getOrElse(Map.empty),
          Option(requestBody.defaultConfigFile)
        )
      ).asJava
    )
  }

  private def normalize(part: String): String = {
    if (PathUtils.isInvalidEncodedLocation(part)) throw new InvalidEnvironmentRequestException("Invalid request")
    Environment.normalize(part)
  }
}
